/***************************************************************************
**
**  This file is part of warangps.
**
**  warangps is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  warangps is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-06-27
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpGuiTools.h>

#include "warangpsVersion.h"
#include "warangpsInstallPath.h"
#include "WaranGpsWidget.h"

PACKAGE_INFO("warangps", WARANGPS)

ApplicationHelp * help();

int main(int argc, char ** argv)
{
  Application a(argc, argv, help);

  // Options
  WaranGpsWidget::Mode mode=WaranGpsWidget::NoGps;
  QString stationFile;
  QString coordinateFormat;
  qint16 port=2974;
  double kmaxThreshold=0.25;
  SamplingParameters frequencySampling;
  frequencySampling.setScaleType(SamplingParameters::Log);
  frequencySampling.setStep(1.025);
  frequencySampling.setRange(0.1, 50);

  // Check arguments
  int i, j=1;
  for(i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-no-gps") {
        mode=WaranGpsWidget::NoGps;
      } else if(arg=="-gps") {
        mode=WaranGpsWidget::Gps;
      } else if(arg=="-format") {
        CoreApplication::checkOptionArg(i, argc, argv);
        coordinateFormat=argv[i];
      } else if(arg=="-kmax-threshold") {
        CoreApplication::checkOptionArg(i, argc, argv);
        kmaxThreshold=CoreApplication::toDouble(i, i-1, argv);
        if(kmaxThreshold<=0.0 || kmaxThreshold>=1.0) {
          App::log(tr("warangps: kmax threshold must be between 0 and 1\n"));
          return 2;
        }
      } else if(arg=="-port") {
        CoreApplication::checkOptionArg(i, argc, argv);
        port=static_cast<short>(CoreApplication::toInt(i, i-1, argv));
      } else if(arg=="-s") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString s(argv[i]);
        bool ok=true;
        frequencySampling.setScaleType(SamplingParameters::convertScaleType(s, ok));
        if(!ok) {
          App::log(tr("warangps: bad scale type '%1'\n").arg(argv[i]));
          return 2;
        }
      } else if(arg=="-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        frequencySampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        frequencySampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        frequencySampling.setStep(CoreApplication::toDouble(i, i-1, argv));
      } else {
        App::log(tr("warangps: bad option %1, see -help\n").arg(argv[i]) );
        return 2;
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j < argc) {
    argv[j]=nullptr;
    argc=j;
  }
  if(argc>2) {
    App::log(tr("No more than one file accepted for station list\n") );
    return 2;
  }
  if(argc==2) {
    stationFile=argv[1];
  }

  a.setStream(new StandardStream(stdout));
  SciFigsGlobal s(true);

  // Create main widget and start gui
  WaranGpsWidget * w=new WaranGpsWidget;
  w->setObjectName("warangps");
  w->setKmaxThreshold(kmaxThreshold);
  w->setFrequencySampling(frequencySampling);

  Settings::getWidget(w);
  w->show();
  w->polish();

  if(mode==WaranGpsWidget::NoGps) {
    if(stationFile.isEmpty()) {
      // No option go to home
      w->tab->setCurrentIndex(0);
    } else {
      w->setCoordinates(stationFile, coordinateFormat);
    }
  } else if(!stationFile.isEmpty()){
    w->setGpsStations(port);
  }
  w->setMode(mode);
  w->arrayCoordinateFactor->setValue(1.0);

  int appReturn=a.exec();
  Settings::setWidget(w);
  delete w;
  return appReturn;
}

ApplicationHelp * help()
{
  TRACE;
  ApplicationHelp * h=new ApplicationHelp;
  h->setOptionSummary( "[OPTIONS] [FILE]" );
  h->setComments( "WARAN Global Positioning System is a graphic interface to monitor an array of WARAN stations equiped "
                  "with GPS for precise positionning. It includes all necessary tools to handle coordinates of stations "
                  "on the field. It computes theoretical array response (FK) and co-array for ring selection (SPAC). "
                  "Coordinates can be loaded from FILE. It replaces the former build_array for array response computation.");
  h->addGroup("warangps", "warangps");
  h->addOption("-no-gps","Starts off-line mode expecting coordinates from a coordinate file (default mode). "
               "If no '-format' is provided, the file parser dialog box is shown for importing FILE.");
  h->addOption("-gps","Starts on-line mode expecting coordinates from GPS devices. FILE is a two-column file: "
                      "station_name station_ip");
  h->addOption("-port PORT","Port used by GPS stations (default=2974).");
  h->addOption("-format FILE","Defines the format of the coordinate file. This is a '.ctparser'' file that can be exported from the "
                              "coordinate parser dialog box.");
  h->addOption("-kmax-threshold VAL","Define threshold for Kmax computation (default=0.25).");
  h->addGroup("Dispersion limits","disp");
  h->addOption("-s <sampling>","Defines the sampling type:\n"
                              "  period     regular sampling in period\n"
                              "  frequency  regular sampling in frequency\n"
                              "  log        regular sampling in log(frequency) (default)");
  h->addOption("-min <min>","Minimum of range for dispersion curve (default=0.2)");
  h->addOption("-max <max>","Maximum of range for dispersion curve (default=50)");
  h->addOption("-step <s>","Step between samples (default=1.025)");
  return h;
}
