/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-11-08
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SamplePlot.h"

/*!
  \class SamplePlot SamplePlot.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

const QString SamplePlot::xmlSamplePlotTag="SamplePlot";

REGISTER_GRAPHCONTENTLAYER(SamplePlot, "SamplePlot" )

/*!
  Description of constructor still missing
*/
SamplePlot::SamplePlot(AxisWindow * parent)
  : GraphContentsLayer(parent)
{
  TRACE;
  _hist=nullptr;
  _dotDiameter=0.1;
  setOpacity(0.05);
}

/*!
  Description of destructor still missing
*/
SamplePlot::~SamplePlot()
{
  TRACE;
}

inline void SamplePlot::drawDot(const GraphContentsOptions& gc, QPainter& p,
                                int pointSize, int halfSize,
                                const Histogram2D::Sample& sample) const
{
  if(sample.isValid()) {
    if(sample.x()>gc.xVisMin() && sample.x()<gc.xVisMax()) {
      if(sample.y()>gc.yVisMin() && sample.y()<gc.yVisMax()) {
        if(sample.category()<_categories->count()) {
          QColor col;
          const Pen& pen=_categories->symbol(sample.category()).pen();
          if(pen.lineStyle()==Pen::SolidLine) {
            col=pen.color();
            p.setPen(col);
            p.setBrush(col);
            int orx=gc.xr2s(sample.x())-halfSize;
            int ory=gc.yr2s(sample.y())-halfSize;
            p.drawEllipse(orx, ory, pointSize, pointSize);
          }
        }
      }
    }
  }
}

void SamplePlot::paintData(const LayerPainterRequest& lp, QPainter& p, double dotpercm) const
{
  TRACE;
  const GraphContentsOptions& gc=lp.options();

  double pointSizeF=_dotDiameter*dotpercm;
  int pointSize, halfSize;
  if(pointSizeF<2.0) {
    pointSize=2;
    halfSize=1;
  } else {
    pointSize=qRound(pointSizeF);
    halfSize=qRound(0.5*pointSizeF);
  }
  int n=_hist->sampleCount();
  for(int i=0; i<n; i++) {
    drawDot(gc, p, pointSize, halfSize, _hist->sample(i));
  }
}

Rect SamplePlot::boundingRect() const
{
  TRACE;
    Rect r;
    int n=_hist->sampleCount();
    for(int i=0; i<n; i++) {
      const Histogram2D::Sample& s=_hist->sample(i);
      r.add(s.x(), s.y());
    }
    return r;
}

void SamplePlot::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
{
  TRACE;
  GraphContentsLayer::xml_writeProperties(s, context);
  XMLSciFigs * scifigsContext=static_cast<XMLSciFigs *>(context);
  if(scifigsContext->data()) writeBinaryData(s, context);
}

void SamplePlot::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
{
  TRACE;
  XMLSciFigs * scifigsContext=static_cast<XMLSciFigs *>(context);
  if(scifigsContext->data()) {
    _categories->xml_save(s, context);
  }
}

void SamplePlot::xml_writeBinaryData(XML_WRITEBINARYDATA_ARGS) const
{
  TRACE;
  Q_UNUSED(context)
  int n=_hist->sampleCount();
  s << n;
  for(int i=0; i<n; i++) {
    s << _hist->sample(i).x();
  }
  for(int i=0; i<n; i++) {
    s << _hist->sample(i).y();
  }
  for(int i=0; i<n; i++) {
    s << _hist->sample(i).category();
  }
}

uint SamplePlot::_tab=PropertyTab::uniqueId();

/*!
  Setup property editor
*/
void SamplePlot::addProperties(PropertyProxy * pp)
{
  TRACE;
  if(pp->setCurrentTab(_tab)) {
    pp->addReference(this);
  } else {
    XYPlotProperties * w=new XYPlotProperties;
    w->removeSymbol();
    pp->addTab(_tab, tr("Sample plot"), w, this);
  }
}

/*!
  Clean property editor
*/
void SamplePlot::removeProperties(PropertyProxy * pp)
{
  TRACE;
  pp->removeTab(_tab, this);
}


void SamplePlot::properties(PropertyWidget * w) const
{
  TRACE;
  w->setValue(XYPlotProperties::DotDiameter, dotDiameter()*10.0);
}

void SamplePlot::setProperty(uint, int pid, QVariant val)
{
  TRACE;
  switch(pid) {
  case XYPlotProperties::DotDiameter:
    setDotDiameter(val.toDouble()*0.1);
    graphContents()->deepUpdate();
    break;
  default:
    break;
  }
}
