/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-16
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "Reader.h"
#include "ClassificationParameters.h"

/*!
  \class Reader Reader.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

Reader::Reader()
  : _relativeRange(0.2),
    _undefinedEll(true),
    _splitEllSign(false),
    _autoPick(false),
    _noiseDev(std::numeric_limits<double>::infinity()),
    _powerDev(std::numeric_limits<double>::infinity()),
    _slownessDev(0.0),
    _minK(0.0),
    _maxK(std::numeric_limits<double>::infinity()),
    _minV(0.0),
    _maxV(std::numeric_limits<double>::infinity()),
    _minEll(-90.0),
    _maxEll(90.0),
    _minVerticalNoise(0.0),
    _maxVerticalNoise(std::numeric_limits<double>::infinity()),
    _minHorizontalNoise(0.0),
    _maxHorizontalNoise(std::numeric_limits<double>::infinity()),
    _minTotalNoise(0.0),
    _maxTotalNoise(std::numeric_limits<double>::infinity()),
    _minDeltaNoise(-std::numeric_limits<double>::infinity()),
    _maxDeltaNoise(std::numeric_limits<double>::infinity()),
    _minSigmaNoise(0.0),
    _maxSigmaNoise(std::numeric_limits<double>::infinity()),
    _minAzimuth(0.0),
    _maxAzimuth(std::numeric_limits<double>::infinity()),
    _minPower(0.0),
    _maxPower(std::numeric_limits<double>::infinity())
{
  _showPlots=Plots();
  _plotType=Histograms;

  _dispersionTitle=tr("Slowness (s/m)");
  _ellipticityTitle=tr("Ellipticity");
  _verticalNoiseTitle=tr("Incoherent noise Rz/N");
  _horizontalNoiseTitle=tr("Incoherent noise Rh/N");
  _totalNoiseTitle=tr("Incoherent noise R/N");
  _deltaNoiseTitle=tr("Incoherent noise Rz/N-Rh/N");
  _sigmaNoiseTitle=tr("Incoherent noise Rh/Rz");
  _azimuthTitle=tr("Azimuth (degrees)");
  _powerTitle=tr("Power");

  _dispersionSampling.setMinimum(1.0/3500.0);
  _dispersionSampling.setMaximum(1.0/50.0);
  _dispersionSampling.setScaleType(SamplingParameters::Log);
  _dispersionSampling.setStep(1.015);
  _ellipticitySampling.setMinimum(-90.0);
  _ellipticitySampling.setMaximum(90.0);
  _ellipticitySampling.setStep(1.0);
  _verticalNoiseSampling.setMinimum(0.01);
  _verticalNoiseSampling.setMaximum(10);
  _verticalNoiseSampling.setScaleType(SamplingParameters::Log);
  _verticalNoiseSampling.setStep(1.05);
  _horizontalNoiseSampling.setMinimum(0.01);
  _horizontalNoiseSampling.setMaximum(10);
  _horizontalNoiseSampling.setScaleType(SamplingParameters::Log);
  _horizontalNoiseSampling.setStep(1.05);
  _totalNoiseSampling.setMinimum(0.01);
  _totalNoiseSampling.setMaximum(10);
  _totalNoiseSampling.setScaleType(SamplingParameters::Log);
  _totalNoiseSampling.setStep(1.05);
  _deltaNoiseSampling.setMinimum(-5.0);
  _deltaNoiseSampling.setMaximum(5.0);
  _deltaNoiseSampling.setScaleType(SamplingParameters::Linear);
  _deltaNoiseSampling.setStep(0.01);
  _sigmaNoiseSampling.setMinimum(0.01);
  _sigmaNoiseSampling.setMaximum(10);
  _sigmaNoiseSampling.setScaleType(SamplingParameters::Log);
  _sigmaNoiseSampling.setStep(1.05);
  _azimuthSampling.setMaximum(360.0);
  _azimuthSampling.setStep(5.0);
  _powerSampling.setMaximum(1.0);
  _powerSampling.setCount(200);
  _powerSampling.setScaleType(SamplingParameters::Log);

  memset(_manual.flags, 0, sizeof(int)*5);
  _samples=nullptr;
  _arrayKmin=0.0;
  _arrayKmax=0.0;

  _misfitType=SlownessMisfit;
}

Reader::~Reader()
{
  // Samples are destroyed in ModeWidgets
}

void Reader::setRayleigh()
{
  _showPlots=Dispersion;
  _showPlots|=Ellipticity;
  _showPlots|=VerticalNoise;
  _showPlots|=HorizontalNoise;
  _showPlots|=TotalNoise;
  _showPlots|=DeltaNoise;
  _showPlots|=SigmaNoise;
  _showPlots|=Azimuth;
  _showPlots|=Power;
  if(!_undefinedEll.isSet()) {
    _undefinedEll.setValue(false);
  }
}

bool Reader::setOptions(int& argc, char ** argv)
{
  TRACE;
  // Check arguments
  int i, j=1;
  for(i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-rayleigh") {
        setRayleigh();
      } else if(arg=="-dots") {
        _plotType=Dots;
      } else if(arg=="-histograms") {
       _plotType=Histograms;
      } else if(arg=="-show-disp") {
        _showPlots|=Dispersion;
      } else if(arg=="-show-ell") {
        _showPlots|=Ellipticity;
      } else if(arg=="-show-vertical-noise") {
        _showPlots|=VerticalNoise;
      } else if(arg=="-show-horizontal-noise") {
        _showPlots|=HorizontalNoise;
      } else if(arg=="-show-total-noise") {
        _showPlots|=TotalNoise;
      } else if(arg=="-show-delta-noise") {
        _showPlots|=DeltaNoise;
      } else if(arg=="-show-sigma-noise") {
        _showPlots|=SigmaNoise;
      } else if(arg=="-show-az") {
        _showPlots|=Azimuth;
      } else if(arg=="-show-pow") {
        _showPlots|=Power;
      // --------------- DISPERSION
      } else if(arg=="-disp-sampling" || arg=="-y-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString s(argv[i]);
        bool ok=true;
        _dispersionSampling.setScaleType(SamplingParameters::convertScaleType(s, ok));
        if(!ok) {
          App::log(tr("Error setting dispersion sampling\n"));
          return false;
        }
        _manual.dispersion.gridScaleType=true;
      } else if(arg=="-disp-min" || arg=="-y-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _dispersionSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.dispersion.minimum=true;
      } else if(arg=="-disp-max" || arg=="-y-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _dispersionSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.dispersion.maximum=true;
      } else if(arg=="-disp-count" || arg=="-y-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _dispersionSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.dispersion.stepCount=true;
        if(_dispersionSampling.count()<=0) {
          App::log(tr("negative or null number of dispersion samples (option -disp-count)\n") );
          return false;
        }
      } else if(arg=="-disp-step" || arg=="-y-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _dispersionSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.dispersion.stepCount=true;
        if(_dispersionSampling.step()<=0) {
          App::log(tr("negative or null step for dispersion samples (option -disp-step)\n") );
          return false;
        }
      } else if(arg=="-disp-title" || arg=="-y-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _dispersionTitle=argv[i];
      // --------------- ELLIPTICITY
      } else if(arg=="-ell-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString s(argv[i]);
        bool ok=true;
        _ellipticitySampling.setScaleType(SamplingParameters::convertScaleType(s, ok));
        if(!ok) {
          App::log(tr("Error setting ellipticity sampling\n"));
          return false;
        }
        _manual.ellipticity.gridScaleType=true;
      } else if(arg=="-ell-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ellipticitySampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.ellipticity.minimum=true;
        _minEll.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-ell-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ellipticitySampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.ellipticity.maximum=true;
        _maxEll.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-ell-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ellipticitySampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.ellipticity.stepCount=true;
        if(_ellipticitySampling.count()<=0) {
          App::log(tr("negative or null number of ellipticity samples (option -ell-count)\n") );
          return false;
        }
      } else if(arg=="-ell-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ellipticitySampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.ellipticity.stepCount=true;
        if(_ellipticitySampling.count()<=0) {
          App::log(tr("negative or null number of ellipticity samples (option -ell-count)\n") );
          return false;
        }
      } else if(arg=="-ell-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ellipticityTitle=argv[i];
      // --------------- VERTICAL NOISE
      } else if(arg=="-vertical-noise-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString s(argv[i]);
        bool ok=true;
        _verticalNoiseSampling.setScaleType(SamplingParameters::convertScaleType(s, ok));
        if(!ok) {
          App::log(tr("Error setting vertical noise sampling\n"));
          return false;
        }
        _manual.verticalNoise.gridScaleType=true;
      } else if(arg=="-vertical-noise-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _verticalNoiseSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.verticalNoise.minimum=true;
        _minVerticalNoise.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-vertical-noise-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _verticalNoiseSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.verticalNoise.maximum=true;
        _maxVerticalNoise.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-vertical-noise-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _verticalNoiseSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.verticalNoise.stepCount=true;
        if(_verticalNoiseSampling.count()<=0) {
          App::log(tr("negative or null number of noise samples (option -vertical-noise-count)\n") );
          return false;
        }
      } else if(arg=="-vertical-noise-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _verticalNoiseSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.verticalNoise.stepCount=true;
        if(_verticalNoiseSampling.count()<=0) {
          App::log(tr("negative or null number of vertical noise samples (option -vertical-noise-count)\n") );
          return false;
        }
      } else if(arg=="-vertical-noise-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _verticalNoiseTitle=argv[i];
      // --------------- HORIZONTAL NOISE
      } else if(arg=="-horizontal-noise-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString s(argv[i]);
        bool ok=true;
        _horizontalNoiseSampling.setScaleType(SamplingParameters::convertScaleType(s, ok));
        if(!ok) {
          App::log(tr("Error setting horizontal noise sampling\n"));
          return false;
        }
        _manual.horizontalNoise.gridScaleType=true;
      } else if(arg=="-horizontal-noise-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _horizontalNoiseSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.horizontalNoise.minimum=true;
        _minHorizontalNoise.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-horizontal-noise-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _horizontalNoiseSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.horizontalNoise.maximum=true;
        _maxHorizontalNoise.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-horizontal-noise-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _horizontalNoiseSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.horizontalNoise.stepCount=true;
        if(_horizontalNoiseSampling.count()<=0) {
          App::log(tr("negative or null number of noise samples (option -horizontal-noise-count)\n") );
          return false;
        }
      } else if(arg=="-horizontal-noise-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _horizontalNoiseSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.horizontalNoise.stepCount=true;
        if(_horizontalNoiseSampling.count()<=0) {
          App::log(tr("negative or null number of noise samples (option -horizontal-noise-count)\n") );
          return false;
        }
      } else if(arg=="-horizontal-noise-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _horizontalNoiseTitle=argv[i];
      // --------------- TOTAL NOISE
      } else if(arg=="-total-noise-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString s(argv[i]);
        bool ok=true;
        _totalNoiseSampling.setScaleType(SamplingParameters::convertScaleType(s, ok));
        if(!ok) {
          App::log(tr("Error setting total noise sampling\n"));
          return false;
        }
        _manual.totalNoise.gridScaleType=true;
      } else if(arg=="-total-noise-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _totalNoiseSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.totalNoise.minimum=true;
        _minTotalNoise.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-total-noise-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _totalNoiseSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.totalNoise.maximum=true;
        _maxTotalNoise.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-total-noise-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _totalNoiseSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.totalNoise.stepCount=true;
        if(_totalNoiseSampling.count()<=0) {
          App::log(tr("negative or null number of noise samples (option -total-noise-count)\n") );
          return false;
        }
      } else if(arg=="-total-noise-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _totalNoiseSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.totalNoise.stepCount=true;
        if(_totalNoiseSampling.count()<=0) {
          App::log(tr("negative or null number of noise samples (option -total-noise-count)\n") );
          return false;
        }
      } else if(arg=="-total-noise-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _totalNoiseTitle=argv[i];
      // --------------- DELTA NOISE
      } else if(arg=="-delta-noise-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString s(argv[i]);
        bool ok=true;
        _deltaNoiseSampling.setScaleType(SamplingParameters::convertScaleType(s, ok));
        if(!ok) {
          App::log(tr("Error setting delta noise sampling\n"));
          return false;
        }
        _manual.deltaNoise.gridScaleType=true;
      } else if(arg=="-delta-noise-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _deltaNoiseSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.deltaNoise.minimum=true;
        _minDeltaNoise.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-delta-noise-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _deltaNoiseSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.deltaNoise.maximum=true;
        _maxDeltaNoise.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-delta-noise-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _deltaNoiseSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.deltaNoise.stepCount=true;
        if(_deltaNoiseSampling.count()<=0) {
          App::log(tr("negative or null number of noise samples (option -delta-noise-count)\n") );
          return false;
        }
      } else if(arg=="-delta-noise-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _deltaNoiseSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.deltaNoise.stepCount=true;
        if(_deltaNoiseSampling.count()<=0) {
          App::log(tr("negative or null number of noise samples (option -delta-noise-count)\n") );
          return false;
        }
      } else if(arg=="-delta-noise-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _deltaNoiseTitle=argv[i];
      // --------------- SIGMA NOISE
      } else if(arg=="-sigma-noise-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString s(argv[i]);
        bool ok=true;
        _sigmaNoiseSampling.setScaleType(SamplingParameters::convertScaleType(s, ok));
        if(!ok) {
          App::log(tr("Error setting sigma noise sampling\n"));
          return false;
        }
        _manual.sigmaNoise.gridScaleType=true;
      } else if(arg=="-sigma-noise-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _sigmaNoiseSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.sigmaNoise.minimum=true;
        _minSigmaNoise.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-sigma-noise-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _sigmaNoiseSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.sigmaNoise.maximum=true;
        _maxSigmaNoise.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-sigma-noise-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _sigmaNoiseSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.sigmaNoise.stepCount=true;
        if(_sigmaNoiseSampling.count()<=0) {
          App::log(tr("negative or null number of noise samples (option -sigma-noise-count)\n") );
          return false;
        }
      } else if(arg=="-sigma-noise-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _sigmaNoiseSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.sigmaNoise.stepCount=true;
        if(_sigmaNoiseSampling.count()<=0) {
          App::log(tr("negative or null number of noise samples (option -sigma-noise-count)\n") );
          return false;
        }
      } else if(arg=="-sigma-noise-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _sigmaNoiseTitle=argv[i];
      // --------------- AZIMUTH
      } else if(arg=="-az-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString s(argv[i]);
        bool ok=true;
        _azimuthSampling.setScaleType(SamplingParameters::convertScaleType(s, ok));
        if(!ok) {
          App::log(tr("Error setting azimuth sampling\n"));
          return false;
        }
        _manual.azimuth.gridScaleType=true;
      } else if(arg=="-az-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _azimuthSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.azimuth.minimum=true;
        _minAzimuth.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-az-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _azimuthSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.azimuth.maximum=true;
        _maxAzimuth.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-az-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _azimuthSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.azimuth.stepCount=true;
        if(_azimuthSampling.count()<=0) {
          App::log(tr("negative or null number of azimuth samples (option -az-count)\n") );
          return false;
        }
      } else if(arg=="-az-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _azimuthSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.azimuth.stepCount=true;
        if(_azimuthSampling.count()<=0) {
          App::log(tr("negative or null number of azimuth samples (option -az-count)\n") );
          return false;
        }
      } else if(arg=="-az-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _azimuthTitle=argv[i];
      // --------------- POWER
      } else if(arg=="-pow-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString s(argv[i]);
        bool ok=true;
        _powerSampling.setScaleType(SamplingParameters::convertScaleType(s, ok));
        if(!ok) {
          App::log(tr("Error setting power sampling\n"));
          return false;
        }
        _manual.power.gridScaleType=true;
      } else if(arg=="-pow-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _powerSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.power.minimum=true;
        _minPower.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-pow-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _powerSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.power.maximum=true;
        _maxPower.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-pow-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _powerSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.power.stepCount=true;
        if(_powerSampling.count()<=0) {
          App::log(tr("negative or null number of power samples (option -pow-count)\n") );
          return false;
        }
      } else if(arg=="-pow-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _powerSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.power.stepCount=true;
        if(_powerSampling.count()<=0) {
          App::log(tr("negative or null number of power samples (option -pow-count)\n") );
          return false;
        }
      } else if(arg=="-pow-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _powerTitle=argv[i];
      } else if(arg=="-curve-rel-range") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _relativeRange.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-ref-disp-layer") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _referenceDispersionLayer=argv[i];
      } else if(arg=="-ref-ell-layer") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _referenceEllipticityLayer=argv[i];
      } else if(arg=="-filter") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _filters.append(argv[i]);
      } else if(arg=="-filter-example") {
        ClassificationParameters  param;
        QTextStream(stdout) << param.AbstractParameters::toString()
                            << param.toString();
        return 0;
      } else if(arg=="-undef-ell") {
        _undefinedEll.setValue(true);
      } else if(arg=="-split-ell-sign") {
        _splitEllSign.setValue(true);
      } else if(arg=="-auto-pick") {
        _autoPick.setValue(true);
      } else if(arg=="-low-noise") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _noiseDev.setValue(CoreApplication::toDouble(i, i-1, argv));
        CoreApplication::checkOptionArg(i, argc, argv);
        double dev=CoreApplication::toDouble(i, i-2, argv);
        if(_slownessDev.isSet() && _slownessDev.value()!=dev) {
          App::log(tr("Slowness deviation cannot have multiple values, defined with another option (-high-power).") );
          return false;
        }
        _slownessDev.setValue(dev);
      } else if(arg=="-high-power") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _powerDev.setValue(CoreApplication::toDouble(i, i-1, argv));
        CoreApplication::checkOptionArg(i, argc, argv);
        double dev=CoreApplication::toDouble(i, i-2, argv);
        if(_slownessDev.isSet() && _slownessDev.value()!=dev) {
          App::log(tr("Slowness deviation cannot have multiple values, defined with another option (-low-noise).") );
          return false;
        }
        _slownessDev.setValue(dev);
      } else if(arg=="-misfit") {
        CoreApplication::checkOptionArg(i, argc, argv);
        if(QString(argv[i]).toLower()=="slowness") {
          _misfitType=SlownessMisfit;
        } else {
          _misfitType=EllipticityMisfit;
        }
        _action=Misfit;
      } else if(arg=="-v-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _minV.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-v-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _maxV.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-k-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _minK.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-k-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _maxK.setValue(CoreApplication::toDouble(i, i-1, argv));
      } else {
        argv[j++]=argv[i];
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j<argc) {
    argv[j]=nullptr;
    argc=j;
  }
  if(_showPlots==0) {
    _showPlots=Dispersion;
    _showPlots|=Azimuth;
    _showPlots|=Power;
  }
  return HistogramReader::setOptions(argc, argv);
}

void Reader::helpY(ApplicationHelp * h)
{
  h->addGroup("Dispersion Values", "disp");
  h->addOption("-disp-sampling <SAMPLING>","Defines the slowness sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inverse    regular sampling on an inverse scale");
  h->addOption("-disp-min <MIN>","Minimum of range for slownness axis (default=deduced from data)");
  h->addOption("-disp-max <MAX>","Maximum of range for slowness axis (default=deduced from data)");
  h->addOption("-disp-step <STEP>","Step between samples along slowness");
  h->addOption("-disp-count <COUNT>","Number of samples along slowness");
  h->addOption("-disp-title <TITLE>","Title for slowness axis");
  h->addGroup("Ellipticity Values", "ell");
  h->addOption("-ell-sampling <SAMPLING>","Defines the ellipticity sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inverse    regular sampling on an inverse scale");
  h->addOption("-ell-min <MIN>","Minimum of range for ellipticity axis (default=deduced from data)");
  h->addOption("-ell-max <MAX>","Maximum of range for ellipticity axis (default=deduced from data)");
  h->addOption("-ell-step <STEP>","Step between samples along ellipticity");
  h->addOption("-ell-count <COUNT>","Number of samples along ellipticity");
  h->addOption("-ell-title <TITLE>","Title for ellipticity axis");
  h->addGroup("Vertical noise Values", "vertNoise");
  h->addOption("-vertical-noise-sampling <SAMPLING>","Defines the vertical noise sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inverse    regular sampling on an inverse scale");
  h->addOption("-vertical-noise-min <MIN>","Minimum of range for vertical noise axis (default=deduced from data)");
  h->addOption("-vertical-noise-max <MAX>","Maximum of range for vertical noise axis (default=deduced from data)");
  h->addOption("-vertical-noise-step <STEP>","Step between samples along vertical noise");
  h->addOption("-vertical-noise-count <COUNT>","Number of samples along vertical noise");
  h->addOption("-vertical-noise-title <TITLE>","Title for vertical noise axis");
  h->addGroup("Horizontal noise Values", "horiNoise");
  h->addOption("-horizontal-noise-sampling <SAMPLING>","Defines the horizontal noise sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inverse    regular sampling on an inverse scale");
  h->addOption("-horizontal-noise-min <MIN>","Minimum of range for horizontal noise axis (default=deduced from data)");
  h->addOption("-horizontal-noise-max <MAX>","Maximum of range for horizontal noise axis (default=deduced from data)");
  h->addOption("-horizontal-noise-step <STEP>","Step between samples along horizontal noise");
  h->addOption("-horizontal-noise-count <COUNT>","Number of samples along horizontal noise");
  h->addOption("-horizontal-noise-title <TITLE>","Title for horizontal noise axis");
  h->addGroup("Total noise Values", "totNoise");
  h->addOption("-total-noise-sampling <SAMPLING>","Defines the total noise sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inverse    regular sampling on an inverse scale");
  h->addOption("-total-noise-min <MIN>","Minimum of range for total noise axis (default=deduced from data)");
  h->addOption("-total-noise-max <MAX>","Maximum of range for total noise axis (default=deduced from data)");
  h->addOption("-total-noise-step <STEP>","Step between samples along total noise");
  h->addOption("-total-noise-count <COUNT>","Number of samples along total noise");
  h->addOption("-total-noise-title <TITLE>","Title for total noise axis");
  h->addGroup("Delta noise Values", "deltaNoise");
  h->addOption("-delta-noise-sampling <SAMPLING>","Defines the delta noise sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inverse    regular sampling on an inverse scale");
  h->addOption("-delta-noise-min <MIN>","Minimum of range for delta noise axis (default=deduced from data)");
  h->addOption("-delta-noise-max <MAX>","Maximum of range for delta noise axis (default=deduced from data)");
  h->addOption("-delta-noise-step <STEP>","Step between samples along delta noise");
  h->addOption("-delta-noise-count <COUNT>","Number of samples along delta noise");
  h->addOption("-delta-noise-title <TITLE>","Title for delta noise axis");
  h->addGroup("Sigma noise Values", "sigmaNoise");
  h->addOption("-sigma-noise-sampling <SAMPLING>","Defines the sigma noise sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inverse    regular sampling on an inverse scale");
  h->addOption("-sigma-noise-min <MIN>","Minimum of range for sigma noise axis (default=deduced from data)");
  h->addOption("-sigma-noise-max <MAX>","Maximum of range for sigma noise axis (default=deduced from data)");
  h->addOption("-sigma-noise-step <STEP>","Step between samples along sigma noise");
  h->addOption("-sigma-noise-count <COUNT>","Number of samples along sigma noise");
  h->addOption("-sigma-noise-title <TITLE>","Title for sigma noise axis");
  h->addGroup("Azimuth Values", "az");
  h->addOption("-az-sampling <SAMPLING>","Defines the azimuth sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inverse    regular sampling on an inverse scale");
  h->addOption("-az-min <MIN>","Minimum of range for azimuth axis (default=deduced from data)");
  h->addOption("-az-max <MAX>","Maximum of range for azimuth axis (default=deduced from data)");
  h->addOption("-az-step <STEP>","Step between samples along azimuth");
  h->addOption("-az-count <COUNT>","Number of samples along azimuth");
  h->addOption("-az-title <TITLE>","Title for azimuth axis");
  h->addGroup("Power Values", "pow");
  h->addOption("-pow-sampling <SAMPLING>","Defines the power sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inverse    regular sampling on an inverse scale");
  h->addOption("-pow-min <MIN>","Minimum of range for power axis (default=deduced from data)");
  h->addOption("-pow-max <MAX>","Maximum of range for power axis (default=deduced from data)");
  h->addOption("-pow-step <STEP>","Step between samples along power");
  h->addOption("-pow-count <COUNT>","Number of samples along power");
  h->addOption("-pow-title <TITLE>","Title for power axis");
}

/*!
  Grab some processing parameter values.
*/
bool Reader::parse(QTextStream& s)
{
  TRACE;
  QString line;
  bool maxFormatHeader=false;
  do {
    line=s.readLine();
    if(line.startsWith("# MAX FORMAT RELEASE")) {
      maxFormatHeader=true;
    } else if(line.startsWith("K_MIN")) {
      _arrayKmin=line.section("=", 1, -1).toDouble();
    } else if(line.startsWith("K_MAX")) {
      _arrayKmax=line.section("=", 1, -1).toDouble();
    } else if(line.startsWith("PROCESS_TYPE")) {
      QString type=line.section("=", 1, -1);
      if(type=="ARDS" ||
         type=="ARTBF" ||
         type=="RTBF" ||
         type=="RTBFCross" ||    // Kept for compatibility
         type=="RTBFSquared" ||
         type=="RTBFRadial" ||
         type=="RTBFVertical" ||
         type=="RDSCross" ||
         type=="RDSSquared" ||
         type=="RDSRadial" ||
         type=="RDSVertical") {
        setRayleigh();
      }
    } else if(line.startsWith("# abs_time frequency slowness azimuth ") ||
              line.startsWith("# abs_time frequency polarization slowness azimuth ") ||  // kept for compatibility
              line.startsWith("# time frequency polarization slowness azimuth ")) {      // kept for compatibility
      if(!configureFKMaxFile(line)) {
        return false;
      }
      _samples->setHeaderLine(headerLine());
      maxFormatHeader=false;
    }
    if(s.atEnd()) {
      break;
    }
  } while(maxFormatHeader || line.startsWith("#"));

  // Effectively load samples, last line is provided because we may reading
  // stdin where seek() is not supported.
  return loadSamples(s, line);
}

/*!

*/
bool Reader::loadSamples(QTextStream& s, const QString& lastLine)
{
  TRACE;
  ConsoleProgress progress;
  progress.setCaption(tr("Loading"));
  Sample sample;
  QString line;
  if(File::isDataLine(lastLine)) {
    if(sample.read(lastLine, 0, _xColumn, _yColumn, _yColumn+1, _yColumn+2,
                   _verticalNoiseColumn, _horizontalNoiseColumn,
                   _powerColumn, _validColumn)) {
      _samples->append(sample);
    } else {
      progress.end(0);
      return false;
    }
  }
  while(!s.atEnd()) {
    if(File::readLine(s, line)) {
      if(sample.read(line, 0, _xColumn, _yColumn, _yColumn+1, _yColumn+2,
                     _verticalNoiseColumn, _horizontalNoiseColumn,
                     _powerColumn, _validColumn)) {
        _samples->append(sample);
      } else {
        return false;
      }
    }
    progress.setValue(_samples->count());
  }
  progress.end(_samples->count());
  // Compatibility with old .max storing plain ellipticity instead of the angular ellipticity
  if(!_angularEllipticity) {
    App::log(tr("Converting old ellipticity format...\n"));
    for(VectorList<Sample>::iterator it=_samples->begin(); it!=_samples->end(); it++) {
      it->convertEllipticity();
    }
  }
  return true;
}

/*!
  Computes default limits from samples
*/
bool Reader::setDefaultLimits()
{
  TRACE;
  if(_samples->isEmpty()) {
    App::log(tr("No samples available\n"));
    return false;
  }
  VectorList<Sample>::iterator it;
  VectorList<double> x;
  for(it=_samples->begin(); it!=_samples->end(); it++) {
    x.append(it->frequency());
  }
  setDefaultXLimits(x);

  double m;
  if(!_manual.dispersion.minimum) {
    m=std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->slowness()<m) {
        m=it->slowness();
      }
    }
    _dispersionSampling.setMinimum(m);
  }
  if(!_manual.dispersion.maximum) {
    m=0.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->slowness()>m) {
        m=it->slowness();
      }
    }
    _dispersionSampling.setMaximum(m);
  }

  if(!_dispersionSampling.isValidRange(true)) {
    App::log(tr("Error while checking dispersion axis\n"));
  }
  _dispersionSampling.includeLimits();

  if(!_manual.ellipticity.minimum) {
    m=std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->ellipticity()<m) {
        m=it->ellipticity();
      }
    }
    _ellipticitySampling.setMinimum(m);
  }
  if(!_manual.ellipticity.maximum) {
    m=-std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->ellipticity()>m) {
        m=it->ellipticity();
      }
    }
    _ellipticitySampling.setMaximum(m);
  }

  if(!_ellipticitySampling.isValidRange(true)) {
    App::log(tr("Error while checking ellipticity axis\n"));
  }
  _ellipticitySampling.includeLimits();

  if(!_manual.verticalNoise.minimum) {
    m=std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->verticalNoise()<m) {
        m=it->verticalNoise();
      }
    }
    if(m<=0.0) {
      m=1e-16;
    }
    _verticalNoiseSampling.setMinimum(m);
  }
  if(!_manual.verticalNoise.maximum) {
    m=0.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->verticalNoise()>m) {
        m=it->verticalNoise();
      }
    }
    _verticalNoiseSampling.setMaximum(m);
  }

  if(!_verticalNoiseSampling.isValidRange(true)) {
    App::log(tr("Back to linear scale for vertical noise axis\n"));
    _verticalNoiseSampling.setScaleType(SamplingParameters::Linear);
  }
  _verticalNoiseSampling.includeLimits();

  if(!_manual.horizontalNoise.minimum) {
    m=std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->horizontalNoise()<m) {
        m=it->horizontalNoise();
      }
    }
    if(m<=0.0) {
      m=1e-16;
    }
    _horizontalNoiseSampling.setMinimum(m);
  }
  if(!_manual.horizontalNoise.maximum) {
    m=0.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->horizontalNoise()>m) {
        m=it->horizontalNoise();
      }
    }
    _horizontalNoiseSampling.setMaximum(m);
  }

  if(!_horizontalNoiseSampling.isValidRange(true)) {
    App::log(tr("Back to linear scale for horizontal noise axis\n"));
    _horizontalNoiseSampling.setScaleType(SamplingParameters::Linear);
  }
  _horizontalNoiseSampling.includeLimits();

  if(!_manual.totalNoise.minimum) {
    m=std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->totalNoise()<m) {
        m=it->totalNoise();
      }
    }
    if(m<=0.0) {
      m=1e-16;
    }
    _totalNoiseSampling.setMinimum(m);
  }
  if(!_manual.totalNoise.maximum) {
    m=0.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->totalNoise()>m) {
        m=it->totalNoise();
      }
    }
    _totalNoiseSampling.setMaximum(m);
  }

  if(!_totalNoiseSampling.isValidRange(true)) {
    App::log(tr("Back to linear scale for total noise axis\n"));
    _totalNoiseSampling.setScaleType(SamplingParameters::Linear);
  }
  _totalNoiseSampling.includeLimits();

  /*if(!_manual.deltaNoise.minimum) {
    m=std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->deltaNoise()<m) {
        m=it->deltaNoise();
      }
    }
    _deltaNoiseSampling.setMinimum(m);
  }
  if(!_manual.deltaNoise.maximum) {
    m=-std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->deltaNoise()>m) {
        m=it->deltaNoise();
      }
    }
    _deltaNoiseSampling.setMaximum(m);
  }*/

  if(!_deltaNoiseSampling.isValidRange(true)) {
    App::log(tr("Back to linear scale for delta noise axis\n"));
    _deltaNoiseSampling.setScaleType(SamplingParameters::Linear);
  }
  _deltaNoiseSampling.includeLimits();

  if(!_manual.sigmaNoise.minimum) {
    m=std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->sigmaNoise()<m) {
        m=it->sigmaNoise();
      }
    }
    if(m<=0.0) {
      m=1e-16;
    }
    _sigmaNoiseSampling.setMinimum(m);
  }
  if(!_manual.sigmaNoise.maximum) {
    m=0.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->sigmaNoise()>m) {
        m=it->sigmaNoise();
      }
    }
    _sigmaNoiseSampling.setMaximum(m);
  }

  if(!_sigmaNoiseSampling.isValidRange(true)) {
    App::log(tr("Back to linear scale for sigma noise axis\n"));
    _sigmaNoiseSampling.setScaleType(SamplingParameters::Linear);
  }
  _sigmaNoiseSampling.includeLimits();

  if(!_manual.azimuth.minimum) {
    m=360.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->azimuth()<m) {
        m=it->azimuth();
      }
    }
    _azimuthSampling.setMinimum(m);
  }
  if(!_manual.azimuth.maximum) {
    m=0.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->azimuth()>m) {
        m=it->azimuth();
      }
    }
    _azimuthSampling.setMaximum(m);
  }

  if(!_azimuthSampling.isValidRange(true)) {
    App::log(tr("Error while checking azimuth axis\n"));
  }
  _azimuthSampling.includeLimits();

  if(!_manual.power.maximum) {
    m=0.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->power()>m) {
        m=it->power();
      }
    }
    _powerSampling.setMaximum(m);
  }
  if(!_manual.power.minimum) {
    _powerSampling.setMinimum(_powerSampling.maximum()*0.001);
  }

  if(!_powerSampling.isValidRange(true)) {
    App::log(tr("Error while checking power axis\n"));
  }
  _powerSampling.includeLimits();

  return true;
}

Histogram2D * Reader::histogram(const SamplingParameters& ySampling) const
{
  TRACE;
  Histogram2D * h=HistogramReader::histogram(ySampling);
  if(h) {
    h->setXTitle(tr("Frequency (Hz)"));
  }
  return h;
}

void Reader::misfit(const QList<Curve<Point2D>>& curves, Curve<Point2D>& m,
                    int * hitCount, double precision) const
{
  App::log(tr("%1 curves loaded, starting misfit on %2 samples...\n")
           .arg(curves.count())
           .arg(_samples->count()));
  Samples::const_iterator it;
  QList<Curve<Point2D>>::const_iterator itc;
  switch(_misfitType) {
  case SlownessMisfit:
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      // Get the curve with the minimum distance to the current sample.
      double minDy=std::numeric_limits<double>::infinity();
      for(itc=curves.begin(); itc!=curves.end(); itc++) {
        const Curve<Point2D>& c=*itc;
        double dy=fabs(1.0-it->slowness()/c.interpole(it->frequency(), LogScale, LogScale).y());
        if(dy<minDy) {
          minDy=dy;
        }
      }
      int i=m.indexOf(it->frequency(), precision);
      Point2D& mp=m.constXAt(i);
      mp.setY(mp.y()+minDy*minDy);
      hitCount[i]++;
    }
    break;
  case EllipticityMisfit:
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      // Get the curve with the minimum distance to the current sample.
      double minDy=std::numeric_limits<double>::infinity();
      for(itc=curves.begin(); itc!=curves.end(); itc++) {
        const Curve<Point2D>& c=*itc;
        double dy=fabs(it->ellipticity()-c.interpole(it->frequency(), LogScale, LinearScale).y());
        if(dy<minDy) {
          minDy=dy;
        }
      }
      int i=m.indexOf(it->frequency(), precision);
      Point2D& mp=m.constXAt(i);
      mp.setY(mp.y()+minDy*minDy);
      hitCount[i]++;
    }
    break;
  }
}

