/***************************************************************************
**
**  This file is part of gpsh.
**
**  gpsh is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpsh is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-12-15
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreWave.h>

#include "ShReader.h"

/*!
  \class ShReader ShReader.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
ShReader::ShReader()
    : ArgumentStdinReader()
{
  _nSamples=100;
  _samplingType=LogScale;
  _nSamples=100;
  _minRange=0.2;
  _maxRange=20.0;
  _qp=0.0;
  _qs=0.0;
  _receiverDepth=0.0;
  _referenceDepth=-1.0;
}

bool ShReader::setOptions(int& argc, char ** argv)
{
  TRACE;
  // Check arguments
  int i, j=1;
  for(i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-s") {
        CoreApplication::checkOptionArg(i, argc, argv);
        if(strcmp(argv[i],"period")==0) {
          _samplingType=InverseScale;
        } else if(strcmp(argv[i],"frequency")==0) {
          _samplingType=LinearScale;
        } else {
          _samplingType=LogScale;
        }
      } else if(arg=="-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _minRange=CoreApplication::toDouble(i, i-1, argv);
        if(_minRange<=0.0) {
          App::log(tr("negative or null value for -min\n") );
          return false;
        }
      } else if(arg=="-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _maxRange=CoreApplication::toDouble(i, i-1, argv);
        if(_maxRange<=0.0) {
          App::log(tr("negative or null value for -max\n") );
          return false;
        }
      } else if(arg=="-qp") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _qp=CoreApplication::toDouble(i, i-1, argv);
        if(_qp<=0.0) {
          App::log(tr("null or negative Qp (option -qp)\n") );
          return false;
        }
      } else if(arg=="-qs") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _qs=CoreApplication::toDouble(i, i-1, argv);
        if(_qs<=0.0) {
          App::log(tr("null or negative Qs (option -qs)\n") );
          return false;
        }
      } else if(arg=="-receiver-depth") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _receiverDepth=CoreApplication::toDouble(i, i-1, argv);
        if(_receiverDepth<0.0) {
          App::log(tr("negative receiver depth (option -receiver depth)\n") );
          return false;
        }
      } else if(arg=="-reference-depth") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _referenceDepth=CoreApplication::toDouble(i, i-1, argv);
      } else if(arg=="-n") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _nSamples=CoreApplication::toInt(i, i-1, argv);
        if(_nSamples<2) {
          App::log(tr("number of samples less than 2 (option -n)\n") );
          return false;
        }
      } else {
        App::log(tr("bad option %1, see -help\n").arg(argv[i]) );
        return false;
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j < argc) {
    argv[j]=nullptr;
    argc=j;
  }
  // Compute common sampling scale
  Curve<Point1D> c;
  c.line(_minRange, _maxRange);
  c.resample(_nSamples, _minRange, _maxRange, _samplingType);
  _x=c.xVector();
  return true;
}

bool ShReader::parse(QTextStream& s)
{
  TRACE;
  QTextStream sOut(stdout);
  Seismic1DModel m;
  QString comments;
  if(!m.fromStream(s, &comments)) {
    return false;
  }
  if(m.layerCount()>0) {
    if(_qp>0.0) {
      m.setQp(_qp);
    }
    if(_qs>0.0) {
      m.setQs(_qs);
    }
    // Check that Qp and Qs are properly set
    for(int i=0;i<m.layerCount(); i++) {
      if(m.qp(i)<=0.0) {
        App::log(tr("negative or null value for Qp at layer %1\n").arg(i) );
      }
      if(m.qs(i)<=0.0) {
        App::log(tr("negative or null value for Qs at layer %1\n").arg(i) );
      }
    }
    sOut << "# SH transfer function\n"
         << comments;
    KennettSH kenn(&m);
    if(_referenceDepth<0.0) {
      for(int i=0; i<_x.count(); i++) {
        double f=_x.at(i);
        sOut << f << " " << kenn.sh(f, _receiverDepth) << "\n";
      }
    } else {
      for(int i=0; i<_x.count(); i++) {
        double f=_x.at(i);
        sOut << f << " " << kenn.sh(f, _receiverDepth)/kenn.sh(f, _referenceDepth) << "\n";
      }
    }
  }
  return true;
}
