/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-12-11
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "GraphicObject.h"
#include "XMLSciFigs.h"
#include "ParallelBands.h"
#include "AxisWindow.h"
#include "GraphContentsLayerFactory.h"
#include "LayerPainterRequest.h"
#include "LayerLocker.h"

namespace SciFigs {

/*!
  \class ParallelBands ParallelBands.h
  \brief A ParallelBands is a layer to plot infinite parallel bands (along X or Y)
  
*/

const QString ParallelBands::xmlParallelBandsTag="ParallelBands";

REGISTER_GRAPHCONTENTLAYER(ParallelBands, "ParallelBands" );

ParallelBands::ParallelBands(AxisWindow * parent) :
    GraphContentsLayer(parent)
{
  TRACE;
  _axisType=XAxis;
}

ParallelBands::~ParallelBands()
{
  TRACE;

}

void ParallelBands::operator=(ParallelBands& o)
{
  TRACE;
  LayerLocker ll(this);
  _axisType=o._axisType;
  _bands=o._bands;
}

void ParallelBands::setAxisType(AxisType a)
{
  TRACE;
  LayerLocker ll(this);
  _axisType=a;
}

void ParallelBands::clear()
{
  TRACE;
  LayerLocker ll(this);
  _bands.clear();
}

void ParallelBands::addBand(const Brush& b, const Pen& p)
{
  TRACE;
  LayerLocker ll(this);
  _bands.append(ParallelBand(b, p));
}

void ParallelBands::addBand(double value, const Pen& p)
{
  TRACE;
  LayerLocker ll(this);
  _bands.append(ParallelBand(value, p));
}

void ParallelBands::addBand(double min, double max, const Brush& b, const Pen& p)
{
  TRACE;
  LayerLocker ll(this);
  _bands.append(ParallelBand(min, max, b, p));
}

void ParallelBands::removeBand(int index)
{
  TRACE;
  LayerLocker ll(this);
  _bands.remove(index);
}

bool ParallelBands::trackRectangle(int, double rx1, double ry1, double rx2, double ry2, Qt::KeyboardModifiers)
{
  TRACE;
  if(_axisType==XAxis) {
    emit bandPicked(rx1, rx2);
  } else {
    emit bandPicked(ry1, ry2);
  }
  return true;
}

void ParallelBands::paintData(const LayerPainterRequest& lp, QPainter& p,double dotpercm) const
{
  TRACE;
  const GraphContentsOptions& gc=lp.options();
  VectorList<ParallelBand>::const_iterator it=_bands.begin();
  if(_axisType==XAxis) {
    int h=lp.size().height();
    for(;it!=_bands.end();++it) {
      if(lp.terminated()) return;
      const ParallelBand& b=*it;
      double min=b.minimum();
      double max=b.maximum();
      if(max<=gc.xVisMin()) continue;
      if(min>=gc.xVisMax()) continue;
      if(min<gc.xVisMin()) min=gc.xVisMin();
      if(max>gc.xVisMax()) max=gc.xVisMax();
      p.setPen(b.pen().qpen(dotpercm));
      p.setBrush(b.brush().qbrush(dotpercm));
      int bleft=gc.xr2s(min);
      int bwidth=gc.xr2s(max)-bleft;
      if(bwidth==0) {
        p.drawLine(bleft, 0, bleft, h);
      } else {
        p.drawRect(bleft, 0, bwidth, h);
      }
    }
  } else {
    int w=lp.size().width();
    for(;it!=_bands.end();++it) {
      if(lp.terminated()) return;
      const ParallelBand& b=*it;
      double min=b.minimum();
      double max=b.maximum();
      if(max<=gc.yVisMin()) continue;
      if(min>=gc.yVisMax()) continue;
      if(min<gc.yVisMin()) min=gc.yVisMin();
      if(max>gc.yVisMax()) max=gc.yVisMax();
      p.setPen(b.pen().qpen(dotpercm));
      p.setBrush(b.brush().qbrush(dotpercm));
      int btop=gc.yr2s(min);
      int bheight=gc.yr2s(max)-btop;
      if(bheight==0) {
        p.drawLine(0, btop, w, btop);
      } else {
        p.drawRect(0, btop, w, bheight);
      }
    }
  }
}

Rect ParallelBands::boundingRect() const
{
  TRACE;
  if(_bands.isEmpty()) {
    return Rect();
  }
  double min=std::numeric_limits<double>::infinity(), max=-std::numeric_limits<double>::infinity();
  VectorList<ParallelBand>::const_iterator it=_bands.begin();
  for(;it!=_bands.end();++it) {
    const ParallelBand& b=*it;
    if(b.minimum()<min) min=b.minimum();
    if(b.maximum()<min) min=b.maximum();
    if(b.minimum()>max) max=b.minimum();
    if(b.maximum()>max) max=b.maximum();
  }
  if(_axisType==XAxis) {
    return Rect(min, -std::numeric_limits<double>::infinity(), max, std::numeric_limits<double>::infinity());
  } else {
    return Rect(-std::numeric_limits<double>::infinity(), min, std::numeric_limits<double>::infinity(), max);
  }
}

void ParallelBands::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
{
  TRACE;
  XMLSciFigs * scifigsContext=static_cast<XMLSciFigs *>(context);
  if(scifigsContext->data()) {
    VectorList<ParallelBand>::const_iterator it=_bands.begin();
    for(;it!=_bands.end();++it) (*it).xml_save(s, context);
  }
}

XMLMember ParallelBands::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  XMLSciFigs * scifigsContext=static_cast<XMLSciFigs *>(context);
  if(scifigsContext->data()) {
    if(tag=="ParallelBand") {
      _bands.resize(_bands.size()+1);
      return XMLMember(&_bands.back());
    }
  }
  return GraphContentsLayer::xml_member(tag, attributes, context);
}

} // namespace SciFigs
