/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-03-19
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpGuiTools.h>

#include "LegendTableDelegate.h"
#include "LegendTable.h"
#include "LegendTableItem.h"
#include "SciFigsGlobal.h"

namespace SciFigs {

/*!
  \class LegendTableDelegate LegendTableDelegate.h
  \brief Brief description of class still missing

  Full description of class still missing
*/


void LegendTableDelegate::setColumnIndexes(const QHash<int,LegendTable::Column> * columnIndexes)
{
  TRACE;
  _columnIndexes=columnIndexes;
}

inline LegendTable::Column LegendTableDelegate::column(int index) const
{
  QHash<int,LegendTable::Column>::const_iterator it=_columnIndexes->find(index);
  if(it!=_columnIndexes->end()) {
    return it.value();
  } else {
    return LegendTable::ColumnNull;
  }
}

QWidget * LegendTableDelegate::createEditor(QWidget * parent, const QStyleOptionViewItem & option,
                                            const QModelIndex & index) const
{
  TRACE;
  switch (column( index.column())) {
  case LegendTable::ColumnNull: return nullptr;
  case LegendTable::ColumnText:
    return createLineEdit(parent);
  case LegendTable::ColumnPenColor:
  case LegendTable::ColumnSymbolPenColor:
  case LegendTable::ColumnSymbolBrushColor:
  case LegendTable::ColumnTextColor:
    return ItemDelegate::createEditor(parent, option, index);
  case LegendTable::ColumnPenWidth:
  case LegendTable::ColumnSymbolSize:
  case LegendTable::ColumnSymbolPenWidth: {
      DoubleSpinBox * w=createDoubleSpinBox(parent);
      w->setSingleStep(0.1);
      w->setMinimum(0.0);
      w->setMaximum(100.0);
      connect(w, SIGNAL(valueChanged(double)), this, SLOT(commitWidget()));
      return w;
    }
  case LegendTable::ColumnPenStyle:
  case LegendTable::ColumnSymbolPenStyle: {
      QComboBox * w=createComboBox(parent);
      connect(w, SIGNAL(activated(int)), this, SLOT(commitWidget()));
      LegendTable::initPenStyles(w);
      return w;
    }
  case LegendTable::ColumnSymbolType: {
      QComboBox * w=createComboBox(parent);
      connect(w, SIGNAL(activated(int)), this, SLOT(commitWidget()));
      LegendTable::initSymbolTypes(w);
      return w;
    }
  case LegendTable::ColumnSymbolBrushStyle: {
      QComboBox * w=createComboBox(parent);
      connect(w, SIGNAL(activated(int)), this, SLOT(commitWidget()));
      LegendTable::initBrushStyles(w);
      return w;
    }
  }
  return nullptr;
}

QSize LegendTableDelegate::sizeHint (const QStyleOptionViewItem & option, const QModelIndex & index) const
{
  TRACE;
  if(index.row()<2) {
    return ItemDelegate::sizeHint(option, index);
  } else {
    switch (column( index.column()) ) {
    case LegendTable::ColumnNull:
    case LegendTable::ColumnText:
      break;
    case LegendTable::ColumnPenWidth:
    case LegendTable::ColumnSymbolSize:
    case LegendTable::ColumnSymbolPenWidth:
      return QSize(70, 0);
    case LegendTable::ColumnPenColor:
    case LegendTable::ColumnSymbolPenColor:
    case LegendTable::ColumnSymbolBrushColor:
    case LegendTable::ColumnTextColor:
      return QSize(30, 0);
    case LegendTable::ColumnPenStyle:
    case LegendTable::ColumnSymbolPenStyle:
      return QSize(30, 0);
    case LegendTable::ColumnSymbolType:
    case LegendTable::ColumnSymbolBrushStyle:
      return QSize(30, 0);
    }
    return ItemDelegate::sizeHint(option, index);
  }
}

void LegendTableDelegate::paint(QPainter * painter, const QStyleOptionViewItem & option, const QModelIndex & index) const
{
  TRACE;
  if(index.row()<2) {
    ItemDelegate::paint(painter, option, index);
  } else {
    QPoint p=option.rect.center();
    switch (column( index.column()) ) {
    case LegendTable::ColumnNull:
    case LegendTable::ColumnText:
    case LegendTable::ColumnPenWidth:
    case LegendTable::ColumnSymbolSize:
    case LegendTable::ColumnSymbolPenWidth:
      ItemDelegate::paint(painter, option, index);
      break;
    case LegendTable::ColumnPenColor:
    case LegendTable::ColumnSymbolPenColor:
    case LegendTable::ColumnSymbolBrushColor:
    case LegendTable::ColumnTextColor: {
        drawBackground(painter, option, index);
        QColor c=index.model()->data(index).value<QColor>();
        painter->fillRect(p.x()-8, p.y()-8, 16, 16 , c);
      }
      break;
    case LegendTable::ColumnPenStyle:
    case LegendTable::ColumnSymbolPenStyle: {
        drawBackground(painter, option, index);
        Pen pen(painter->pen().color(), 0.3);
        pen.setLineStyle(LegendTable::penStyle(index.model()->data(index).toInt()));
        painter->setPen(pen.qpen(SciFigsGlobal::screenResolution()));
        painter->drawLine(p.x()-16, p.y(), p.x()+16, p.y());
      }
      break;
    case LegendTable::ColumnSymbolType: {
        drawBackground(painter, option, index);
        Symbol::Type t=(Symbol::Type) index.model()->data(index).toInt();
        Symbol s(t, 3.5, painter->pen().color());
        s.paint(*painter, QPointF(p.x()-1, p.y()-1));
      }
      break;
    case LegendTable::ColumnSymbolBrushStyle: {
        drawBackground(painter, option, index);
        Qt::BrushStyle s=LegendTable::brushStyle(index.model()->data(index).toInt());
        painter->setPen(QPen( Qt::NoPen));
        painter->setBrush(QBrush(painter->pen().color(), s) );
        painter->drawRect(p.x()-8, p.y()-8, 16, 16);
      }
      break;
    }
  }
}

void LegendTableDelegate::colorClicked(const QModelIndex& index)
{
  TRACE;
  if(index.row()>1) {
    switch (column(index.column())) {
    case LegendTable::ColumnNull:
    case LegendTable::ColumnText:
    case LegendTable::ColumnPenWidth:
    case LegendTable::ColumnSymbolSize:
    case LegendTable::ColumnSymbolPenWidth:
    case LegendTable::ColumnPenStyle:
    case LegendTable::ColumnSymbolPenStyle:
    case LegendTable::ColumnSymbolType:
    case LegendTable::ColumnSymbolBrushStyle:
      break;
    case LegendTable::ColumnPenColor:
    case LegendTable::ColumnSymbolPenColor:
    case LegendTable::ColumnSymbolBrushColor:
    case LegendTable::ColumnTextColor: {
        QColor c=index.model()->data(index).value<QColor>();
        c=QColorDialog::getColor(c, Application::activeWindow());
        if(c.isValid()) {
          const_cast<QAbstractItemModel *>(index.model())->setData(index, c);
          emit dataChanged(index, StringDiff());
        }
      }
      break;
    }
  }
}

} // namespace SciFigs
