/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-04-18
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef GRAPHCONTENT_H
#define GRAPHCONTENT_H

#include "SciFigsDLLExport.h"
#include "Axis.h"
#include "GraphContentsOptions.h"
#include "GraphContentsLayer.h"
#include "PaintProgress.h"

namespace SciFigs {

  /* Current timing results:

    NVIDIA accelerated device

    For intensive resizing, QPixmap and QImage have the same efficiency (job is distributed differently between
    X11 and application, but global load is the same.
    For intensive repaint, draw operation took 15 ms for QImage while QPixmap is 0. QPixmap waste time in SetPixmap() (13ms).

    ATI Radeon Mobility X1600, with experimental X11 driver radeonhd without acceleted graphics

    QPixmap is rather heavy on resize...
  */
  //#define GRAPHCONTENT_PIXMAP

  class CoordTip;
  class LayerMouseTracking;
  class MouseTracking;
  class LayerPainter;
  class PaintProgress;

  typedef QMap<GraphContentsLayer *, QImage> LayerImages;

  class SCIFIGS_EXPORT GraphContents : public QWidget, public PropertyItem, public XMLClass
  {
    Q_OBJECT
    // Overload to mark the beginning of properties to save to XML
    Q_PROPERTY(QString objectName READ objectName WRITE setObjectName STORED true)
    // Kept only to avoid warning on loading old files
    Q_PROPERTY(QString name READ dummyPropertyString WRITE setObjectName STORED false)
    Q_PROPERTY(double printLineWeight READ dummyPropertyDouble WRITE setLineWeights STORED false)

    Q_PROPERTY(bool gridLines READ gridLines WRITE setGridLines SCRIPTABLE true)
    Q_PROPERTY(QColor gridLineColor READ gridLineColor WRITE setGridLineColor SCRIPTABLE true)
    Q_PROPERTY(bool printBitmap READ printBitmap WRITE setPrintBitmap SCRIPTABLE true)
    Q_PROPERTY(bool transparentMask READ transparentMask WRITE setTransparentMask SCRIPTABLE true)
    Q_PROPERTY(double gridLineWeight READ gridLineWeight WRITE setGridLineWeight SCRIPTABLE true)
    Q_PROPERTY(double contourWeight READ contourWeight WRITE setContourWeight SCRIPTABLE true)
    DUMMY_PROPERTIES
  public:
    GraphContents(AxisWindow * parent);
    ~GraphContents();

    const QString& xml_tagName() const {return xmlGraphContentsTag;}
    bool xml_isValidTagName(const QString& t) const;
    static const QString xmlGraphContentsTag;

    virtual void polish();

    void appendLayer(GraphContentsLayer * layer);
    void prependLayer(GraphContentsLayer * layer);
    void removeLayer(GraphContentsLayer * layer);
    void moveLayerUp(GraphContentsLayer * layer);
    void moveLayerDown(GraphContentsLayer * layer);
    bool mergeLayers();

    AxisWindow * graph() const {return reinterpret_cast<AxisWindow *>(parent());}

    const GraphContentsOptions& options() const {return _options;}

    Scale& scaleX() {return _options.scaleX();}
    Scale& scaleY() {return _options.scaleY();}
    const Scale& constScaleX() const {return _options.scaleX();}
    const Scale& constScaleY() const {return _options.scaleY();}

    void paintMask(QPainter& p);
    void print (QPainter& p, double dotpercm, int w, int h, bool mask);

    bool gridLines() const {return _options.gridLines();}
    void setGridLines(bool gl) {_options.setGridLines(gl);}

    const QColor& gridLineColor() const {return _options.gridLineColor();}
    void setGridLineColor(const QColor& c) {_options.setGridLineColor(c);}

    double gridLineWeight() const {return _options.gridLineWeight();}
    void setGridLineWeight(double lw) {_options.setGridLineWeight(lw);}

    double contourWeight() const {return _options.contourWeight();}
    void setContourWeight(double lw) {_options.setContourWeight(lw);}

    void setLineWeights(double lw);  // kept for compatibility

    bool transparentMask() const {return _options.transparentMask();}
    void setTransparentMask(bool tm) {_options.setTransparentMask(tm);}

    bool printBitmap() const {return _options.printBitmap();}
    void setPrintBitmap(bool sbmp) {_options.setPrintBitmap(sbmp);}

  #ifdef GRAPHCONTENT_PIXMAP
    QPixmap * pixmap() {if(_pixmap.isNull()) return nullptr; else return &_pixmap;}
  #else
    QImage * pixmap() {if(_pixmap.isNull()) return nullptr; else return &_pixmap;}
  #endif
    QAction * action(QString actionName);
    QAction * addAction(QString actionName);
    void zoomIn (const QPoint& c);
    Rect boundingRect() const;
    void swapAxes() {_options.swapAxes();}

    static void init();
    static void clear();
    QString coordinateTipText(const QPoint& mousePos);
    void delayPainting();

    inline void setProgressMaximum(int value);
    inline void setProgressValue(int value);
    void setProgressEnd();

    void addProperties(PropertyProxy * pp);
    void addProperties(GraphContentsLayer * layer);
    void removeProperties(PropertyProxy * pp);
    void properties(PropertyWidget * w) const;
    void setProperty(uint wid, int pid, QVariant val);
    void setProperty(const QVariant& val, const QMetaProperty& p);

    int layerCount() const {return _layers.count();}
    GraphContentsLayer * findLayer(QString tagName, QString layerName=QString(), int startIndex=0);
    template <class T> QList<T *> findLayers(QString layerName);
    void selectLayers(QString layerName);
    void selectAllLayers(bool s);
    bool hasSameLayers(const GraphContents& o) const;

    int indexOf(GraphContentsLayer * layer) {return _layers.indexOf(layer);}
    const GraphContentsLayer * layer(int index) const {return _layers.at(index);}
    Q_INVOKABLE GraphContentsLayer * layer(int index);

    void wheelEvent(QWheelEvent * e);
  public slots:
    void zoom();
    void zoomIn();
    void browse();
    void saveLayers(QString fileName=QString(),
                    const QString& layerName=QString());
    void appendLayers(QString fileName=QString());
    void prependLayers(QString fileName=QString());
    void copyLayers();
    void pasteLayers();
    void copyTipLabel();
    void deepUpdate(int fromLayer=0);
    void showProperties();
  private:
    friend class GraphContentsLayer;
    friend class LayerPropertiesItem;
    friend class CoordTip;
    // Events Handlers
    bool event(QEvent * e);
    void keyPressEvent(QKeyEvent * e);
    void keyReleaseEvent(QKeyEvent * e);
    void mouseDoubleClickEvent(QMouseEvent * e);
    void mousePressEvent(QMouseEvent * e);
    void mouseReleaseEvent(QMouseEvent * e);
    void mouseMoveEvent(QMouseEvent * e);
    void leaveEvent(QEvent *);
    void enterEvent(QEnterEvent *);
    void paintEvent(QPaintEvent * e);
    void resizeEvent(QResizeEvent *);

    void setDefaultCursor();
    void addActions();

    enum TrackingModes{Zoom=0, Browse};
    void beginMouseTracking(const LayerMouseTracking& layer);
    void endMouseTracking(GraphContentsLayer * layer, int id);
    bool isMouseTracking(const GraphContentsLayer *layer, int id) const;
    MouseTracking * _mouseTracking;

    // List of all layers
    QList<GraphContentsLayer *> _layers;

    static CoordTip * _coordTip;

    // Paint operations
    void resizePixmap();
  #ifdef GRAPHCONTENT_PIXMAP
    QPixmap _pixmap;
  #else
    QImage _pixmap;
  #endif
    LayerImages _layerImages;
    QElapsedTimer _paintEventTiming;
    QTimer _paintEventDelay;
    int _paintDepth;
    static LayerPainter * _layerPainter;
    PaintProgress * _progress;

    // Options
    GraphContentsOptions _options;

    // Smoother mouse move engine
    QPoint _mouseMovePoint;
    Qt::MouseButtons _mouseMoveButtons;
    Qt::KeyboardModifiers _mouseMoveModifiers;
    QTimer _mouseMoveDelay;
    static QCursor * _defaultCursor;

    static uint _categoryGraphic, _categoryLayers, _tabGraphicFormat, _tabGraphicLayers;
  protected:
    void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
    void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
    XMLMember xml_member(XML_MEMBER_ARGS);
    void xml_polishChild(XML_POLISHCHILD_ARGS);
    bool xml_polish(XML_POLISH_ARGS);
    bool xml_setProperty(XML_SETPROPERTY_ARGS);
  private slots:
    void delayedMouseMoveEvent();
    void setPixmap(QImage image, LayerImages layerImages);
    void paintProgress();
  signals:
    void zoomIn(double, double, double, double);
    void zoomOut();
    void browse(QRect& r);
    void mouseTrackingBegin();
    void mouseTrackingEnd();
    void mouseMoved(QPoint, Qt::MouseButtons, Qt::KeyboardModifiers);
    void mouseInside(bool);
  };

  inline void GraphContents::setProgressMaximum(int value)
  {
    _progress->setMaximum(value);
  }

  inline void GraphContents::setProgressValue(int value)
  {
    _progress->setValue(value);
  }

  template <class T>
  QList<T *> GraphContents::findLayers(QString layerName)
  {
    TRACE;
    QList<T *> layers;
    int nLayers=_layers.count();
    for(int i=0; i<nLayers; i++) {
      T * layer=qobject_cast<T *>(_layers.at(i));
      if(layer && (layerName.isEmpty() || layer->objectName()==layerName)) {
        layers.append(layer);
      }
    }
    return layers;
  }

} // namespace SciFigs

#endif // GRAPHCONTENT_H
