/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-05-07
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef COLORMAPWIDGET_H
#define COLORMAPWIDGET_H

#include "GraphicObject.h"
#include "Axis.h"

namespace SciFigs {

  class ColorMapProperties;
  class PaletteEditor;
  class ColorMapLayer;

  class SCIFIGS_EXPORT ColorMapWidget : public GraphicObject
  {
    Q_OBJECT
    // Compatibility
    Q_PROPERTY(QString numType READ dummyPropertyString WRITE setNumberType STORED false)
    Q_PROPERTY(int numPrec READ dummyPropertyInt WRITE setNumberPrecision STORED false)
    Q_PROPERTY(QString title READ dummyPropertyString WRITE setTitle STORED false)
    Q_PROPERTY(QString numberType READ dummyPropertyString WRITE setNumberType STORED false)
    Q_PROPERTY(int numberPrecision READ dummyPropertyInt WRITE setNumberPrecision STORED false)
    Q_PROPERTY(QString font READ dummyPropertyString WRITE setFontString STORED false)
    Q_PROPERTY(double lineWeight READ dummyPropertyDouble WRITE setLineWeight STORED false)
    Q_PROPERTY(double tickLength READ dummyPropertyDouble WRITE setTickSize STORED false)
    Q_PROPERTY(double tickLenght READ dummyPropertyDouble WRITE setTickSize STORED false)
  public:
    ColorMapWidget(QWidget *parent=nullptr, Qt::WindowFlags f=Qt::Widget);
    ~ColorMapWidget();

    ColorMapWidget& operator=(const ColorMapWidget& o);

    const QString& xml_tagName() const {return xmlColorMapWidgetTag;}
    static const QString xmlColorMapWidgetTag;

    void setAxis(const Axis& a);
    const Axis& axis() const;

    const ColorMap& colorMap() const;

    void generate(int imin, int imax, const QString& model=ColorPalette::defaultSequential,
                  ColorPalette::Options options=ColorPalette::Continuous, quint16 transparency=COLOR_COMPONENT_MAX);
    Q_INVOKABLE void generate(int imin, int imax, const QString& model,
                              const QString& options, quint16 transparency=COLOR_COMPONENT_MAX);
    Q_INVOKABLE void toGray();

    void setOrientation(Axis::Orientation o) {axis().setOrientation(o);}
    Axis::Orientation orientation() const {return axis().orientation();}

    virtual void addProperties(PropertyProxy * pp);
    virtual void removeProperties(PropertyProxy * pp);
    virtual void properties(PropertyWidget * w) const;
    virtual void setProperty(uint wid, int pid, QVariant val);

    void scaleFonts(FontScales& orignal, double scale);
    void restoreScaleFonts(const FontScales& orignal);
    virtual void print(QPainter& p, double dotpercm, int x0Sheet, int y0Sheet, bool mask);
  public slots:
    void update();
    void deepUpdate();
    void setValues(double min, double max, SamplingParameters::ScaleType t);
    void load(QString fileName);
    void save(QString fileName);
    void setColorMap(const ColorMap& map);
    virtual void showProperties() {GraphicObject::showProperties(_category);}

    void setScaleType(Scale::Type t) {axis().setScaleType(t);}
    void setMajorTicks(double m) {axis().setMajorTicks(m);}
    void setMinorTicks(double m) {axis().setMinorTicks(m);}
    void setNumberType(QString t) {axis().setNumberType(t);}
    void setNumberPrecision(int p) {axis().setCurrentNumberPrecision(p);}
    void setTitle(QString s) {axis().setCurrentTitle(s);}
    void setFontString(QString fs) {axis().setFontString(fs);}
    void setLineWeight(double val) {axis().setLineWeight(val);}
    void setTickSize(double val) {axis().setTickSize(val);}
  protected:
    bool eventFilter(QObject * obj, QEvent * e);
    void resizeEvent(QResizeEvent * );
  private slots:
    void setOrientation();
  signals:
    void changed(ColorMap pal);
  protected:
    XMLMember xml_member(XML_MEMBER_ARGS);
    void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
    void xml_polishChild(XML_POLISHCHILD_ARGS);
  private:
    Axis& axis();
    QMouseEvent * convertMouseEvent(QObject * obj, QEvent * e, QWidget * dest);

    ColorMapLayer * _colorMapLayer;
    static uint _category, _tabColorMap;
  };

} // namespace SciFigs

#endif // COLORMAPWIDGET_H
