/***************************************************************************
**
**  This file is part of QGpGuiTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2010-12-07
**  Copyright: 2010-2019
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QtNetwork>

#include <QGpCoreTools.h>

#include "HttpAccess.h"

namespace QGpGuiTools {

  /*
    Download .pem file with Firefox
    dpwin2lin ~/Downloads/geopsy.pem
    cat ~/Downloads/geopsy.pem | awk '{printf("    \"%s\\n\"\n", $0)}'
  */
  static const char * geopsyCertificate20220830=
      "-----BEGIN CERTIFICATE-----\n"
      "MIIIJTCCBg2gAwIBAgIQJ/bQoqNzq+CKmnkqt7xZNjANBgkqhkiG9w0BAQwFADBE\n"
      "MQswCQYDVQQGEwJOTDEZMBcGA1UEChMQR0VBTlQgVmVyZW5pZ2luZzEaMBgGA1UE\n"
      "AxMRR0VBTlQgT1YgUlNBIENBIDQwHhcNMjIwODMwMDAwMDAwWhcNMjMwODMwMjM1\n"
      "OTU5WjB+MQswCQYDVQQGEwJGUjEeMBwGA1UECAwVQXV2ZXJnbmUtUmjDtG5lLUFs\n"
      "cGVzMSMwIQYDVQQKDBpVbml2ZXJzaXTDqSBHcmVub2JsZSBBbHBlczEqMCgGA1UE\n"
      "AxMhaXN0LWdlb3BzeS51bml2LWdyZW5vYmxlLWFscGVzLmZyMIIBIjANBgkqhkiG\n"
      "9w0BAQEFAAOCAQ8AMIIBCgKCAQEArwEd2ivKQv/x/m4b8wVPd0IiVM4Hg1r3odHE\n"
      "mZemkcTVTR0Vrwrg6jaMmanlF4V7SAaFhBjH6sRiJE0nurg/yH16X01qX3q0LS94\n"
      "bx6KArQV0Wt6PUvqPND+FFpgrvXyKPQIKpO8V00HE6Qe5v22WOG2rup+bmxbSZ12\n"
      "HpgbrBN5Z76XCJLqBoQhZxXw00X4YedOmdG5Ge7TREbZOeImGugCxAZr7njUrwLl\n"
      "om+K/AaiYfgy5ZJ++ILtGSjsZNLk79KJgjeyiuh/0Pwv8S+eUxoRZdQj8RcgfJo9\n"
      "a02kn5J5/TtG1wZSNhmpD8Q6rFGzjG5dj0RVU1A7sbNe3HXvHwIDAQABo4ID1zCC\n"
      "A9MwHwYDVR0jBBgwFoAUbx01SRBsMvpZoJ68iugflb5xegwwHQYDVR0OBBYEFJoI\n"
      "p2nqD2LJtnjxfrlwZV1zjlEBMA4GA1UdDwEB/wQEAwIFoDAMBgNVHRMBAf8EAjAA\n"
      "MB0GA1UdJQQWMBQGCCsGAQUFBwMBBggrBgEFBQcDAjBJBgNVHSAEQjBAMDQGCysG\n"
      "AQQBsjEBAgJPMCUwIwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5jb20vQ1BT\n"
      "MAgGBmeBDAECAjA/BgNVHR8EODA2MDSgMqAwhi5odHRwOi8vR0VBTlQuY3JsLnNl\n"
      "Y3RpZ28uY29tL0dFQU5UT1ZSU0FDQTQuY3JsMHUGCCsGAQUFBwEBBGkwZzA6Bggr\n"
      "BgEFBQcwAoYuaHR0cDovL0dFQU5ULmNydC5zZWN0aWdvLmNvbS9HRUFOVE9WUlNB\n"
      "Q0E0LmNydDApBggrBgEFBQcwAYYdaHR0cDovL0dFQU5ULm9jc3Auc2VjdGlnby5j\n"
      "b20wggF/BgorBgEEAdZ5AgQCBIIBbwSCAWsBaQB2AK33vvp8/xDIi509nB4+GGq0\n"
      "Zyldz7EMJMqFhjTr3IKKAAABgu27JYgAAAQDAEcwRQIhAMsc0qJ6SJNi64PwWbEM\n"
      "s/f6wwuhosbWKAMW3UuN/jBmAiBHvNEHQ0XC7xjgFBsRSSrYLmNpIg/WtT+FsWCs\n"
      "HyFuLgB2AHoyjFTYty22IOo44FIe6YQWcDIThU070ivBOlejUutSAAABgu27JVIA\n"
      "AAQDAEcwRQIgOkmOSIEITfA2uOa+SWX+3bBcvB+LqdzmLK/1oR9ZSMoCIQDZuXqp\n"
      "R6i2FPYklktAbWmw9xC7gBgR08VQxYW7e5tZ+AB3AOg+0No+9QY1MudXKLyJa8kD\n"
      "08vREWvs62nhd31tBr1uAAABgu27JScAAAQDAEgwRgIhAOCrSknNsRkyfvCw3FSj\n"
      "G40LGpy8+jmltcO1TV6QwKWzAiEA3iP4/lMxJaWsdH/6vPZ3i2hSiaqQ+tbwxYJm\n"
      "WO3Yt2Ywgc0GA1UdEQSBxTCBwoIhaXN0LWdlb3BzeS51bml2LWdyZW5vYmxlLWFs\n"
      "cGVzLmZygg5nZW9wc3kub3N1Zy5mcoIOZ2VvcHN5LnUtZ2EuZnKCFmdlb3BzeS51\n"
      "amYtZ3Jlbm9ibGUuZnKCF2ludGVycGFjaWZpYy5nZW9wc3kub3JnghJpc3QtZ2Vv\n"
      "cHN5LnUtZ2EuZnKCFW5lcmllcy1qcjQuZ2VvcHN5Lm9yZ4IRc2VzYW1lLmdlb3Bz\n"
      "eS5vcmeCDnd3dy5nZW9wc3kub3JnMA0GCSqGSIb3DQEBDAUAA4ICAQBaOJxjzKnB\n"
      "46nFgGw1PKcD6uMNvf2427oeTXN3OS+su5xpNI8O7uSFd3AhSa27saWp5aXiJWWY\n"
      "MH3KLPaU8nyXX1oVRbTn3CAuR1Zjz5t6jdOe+Bhn8uLgS4ZXRIMhLIrV5LkvwZB5\n"
      "BXf02vukdq6E6DPZZClWcaG94mLmN8HglBAyi6HDgoOOMaWB4ZBKtTcveVCpRfh8\n"
      "RUx1ezw97IzzGO0+VzctasmVgocgR2sQBt6uupRzkstJDjRcHkh1TQ5xRZr/JFDx\n"
      "jaXsIFBC5G51jeIaRPbVhq1zU0dvkYewXCdyxTeaDqk0RtKVRRnyeGFx26QopKEZ\n"
      "gJYtGfBn7J9pESihLf8jh6sV5hcpK/vhhbpdjceRiagDU4CLQOC4laIGd3sewDZF\n"
      "MroqXSsfZwQv7uHURNj25ViWJH1SNbyAguEU2DLI8ynQ8CagFMoKGvVfwpDqIM2/\n"
      "47O62gmOoOR/XLK8iJ0PTyo6oRcmm7mR1Rj6KM0aynJF60iKnVWriwIG2kXC7DkF\n"
      "klmyQDf+d+gGkGkzD6WtgXcevpyoMFK0EvMVh28v3NAEq86SEpPrNZf/Y1IbKfeW\n"
      "cKlsE+9EcxhvpFgJovhYAPRBLiISO7YqD4FsbBPTACpFciFaiijmU6ndABmila5O\n"
      "G/aDpPk4hSH+hxq9WGJECzA1laxdpolCAg==\n"
      "-----END CERTIFICATE-----\n";

  /*!
    \class HttpAccess HttpAccess.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  HttpAccess::HttpAccess(QObject * parent)
    : QObject(parent)
  {
    TRACE;
  }

  bool HttpAccess::isAvailable()
  {
    TRACE;
    QList<QNetworkInterface> list=QNetworkInterface::allInterfaces();
    for(QList<QNetworkInterface>::iterator it=list.begin(); it!=list.end(); it++) {
      if(it->isValid() && it->flags() & QNetworkInterface::IsRunning) {
        if(it->type()==QNetworkInterface::Ethernet || it->type()==QNetworkInterface::Wifi) {
          return true;
        }
      }
    }
    return false;
  }

  void HttpAccess::configureReply(QNetworkReply * reply)
  {
    // Expired certificate on Aug 30 2023
    QSslCertificate cert(QByteArray(geopsyCertificate20220830), QSsl::Pem);
    QSslError error(QSslError::CertificateExpired, cert);
    QList<QSslError> expectedSslErrors;
    expectedSslErrors.append(error);
    reply->ignoreSslErrors(expectedSslErrors);
    connect(reply, SIGNAL(sslErrors(const QList<QSslError>&)),
            this, SLOT(sslErrors(const QList<QSslError>&)));
    connect(reply, SIGNAL(finished()), this, SLOT(requestFinished()));
  }

  void HttpAccess::get(const QUrl& url)
  {
    TRACE;
    APP_LOG(100, tr("Sending network request...\n  %1\n").arg(url.toString()));
    QNetworkAccessManager * manager=new QNetworkAccessManager(this);
    QNetworkRequest req(url);
    QNetworkReply * reply=manager->get(req);
    configureReply(reply);
  }

  void HttpAccess::post(const QUrl& url, const QByteArray& data)
  {
    TRACE;
    QNetworkAccessManager * manager=new QNetworkAccessManager(this);
    QNetworkRequest req(url);
    req.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply * reply=manager->post(req, data);
    configureReply(reply);
  }

  void HttpAccess::requestFinished()
  {
    TRACE;
    APP_LOG(100, tr("Request finished\n"));
    QNetworkReply * reply=static_cast<QNetworkReply *>(sender());
    if(reply->error()==QNetworkReply::NoError) {
      _receivedData=reply->readAll();
      APP_LOG(100, _receivedData);
      reply->manager()->deleteLater();
      emit finished(true);
    } else {
      APP_LOG(100, reply->errorString());
      emit finished(false);
    }
  }

  void HttpAccess::sslErrors(const QList<QSslError>& errs)
  {
    for(QList<QSslError>::const_iterator it=errs.begin(); it!=errs.end(); it++) {
      APP_LOG(100, tr("SSL error: %1\n").arg(it->errorString()));
    }
  }

} // namespace QGpGuiTools
