/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  QGpCoreTools is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreTools is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-05-20
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "StringDiff.h"
#include "Trace.h"
#include "Global.h"
#include "CoreApplication.h"

namespace QGpCoreTools {

  inline QString StringDiff::Action::beforeContext(const QString& s, int startIndex)
  {
    return s.mid(0, startIndex);
  }

  inline QString StringDiff::Action::afterContext(const QString& s, int endIndex)
  {
    return s.mid(endIndex);
  }

  /*!
  */
  StringDiff::Action::Action()
  {
    _startIndex=0;
    _endIndex=0;
  }

  /*!
  */
  StringDiff::Action::Action(const QString& s, int startIndex, int count, const QString& insert)
  {
    _startIndex=startIndex;
    _endIndex=_startIndex+count;

    _beforeText=s.mid(0, _startIndex);
    _afterText=s.mid(_endIndex);
    _insertText=insert;
  }

  void StringDiff::Action::setContextStartIndex(int minStartIndex)
  {
    int d=minStartIndex-(_startIndex-_beforeText.size());
    if(d>0) {
      //_beforeText=_beforeText.mid(d);
    }
  }

  QString StringDiff::Action::toString() const
  {
    QString str;
    if(_insertText.isEmpty()) {
      str="[Remove] ";
    } else {
      str="[Insert] ";
    }
    str+=QString("between '%1' and '%2' set '%3', start=%4, end=%5\n")
        .arg(_beforeText)
        .arg(_afterText)
        .arg(_insertText)
        .arg(_startIndex)
        .arg(_endIndex);
    return str;
  }

  /*!
    \class StringDiff StringDiff.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  inline bool StringDiff::atEnd(const QString& s1, const QString& s2, int i1, int i2)
  {
    return i1>=s1.size() || i2>=s2.size();
  }

  void StringDiff::setDifferences(const QString& s1, const QString& s2)
  {
    _actions.clear();
    int i1=0, i2=0;
    int i1p=0, i2p=0, i1m=0, i2m=0;
    while(!atEnd(s1, s2, i1, i2)) {
      int nr=0;
      while(!atEnd(s1, s2, i1, i2)) {

        i1p=i1;
        i1m=0; // Number of characters in common
        i2p=i2;
        i2m=0; // Number of characters in common
        while(!atEnd(s1, s2, i1p, i2)) {
          if(s1[i1p]==s2[i2]) {
            i1m=1;
            while(!atEnd(s1, s2, i1p+i1m, i2+i1m) && s1[i1p+i1m]==s2[i2+i1m]) {
              i1m++;
            }
            break;
          }
          i1p++;
        }

        if(i1<i1p) { // Some initial different characters, further testing is required
          while(!atEnd(s1, s2, i1, i2p)) {
            if(s1[i1]==s2[i2p]) {
              i2m=1;
              while(!atEnd(s1, s2, i1+i2m, i2p+i2m) && s1[i1+i2m]==s2[i2p+i2m]) {
                i2m++;
              }
              break;
            }
            i2p++;
          }
        }

        if(i1m>=3 || i2m>=3 || (i1m>0 && i2m==0) || (i1m==0 && i2m>0)) {
          break;
        } else {
          i1++;
          i2++;
          nr++;
        }
      }
      if(s1.mid(i1-nr, nr)!=s2.mid(i2-nr, nr)) {
        add(Action(s1, i1-nr, nr, s2.mid(i2-nr, nr)));
      }
      if(i1m>i2m) {
        if(i1p>i1) {
          add(Action(s1, i1, i1p-i1));
        }
        i1=i1p+i1m;
        i2+=i1m;
      } else {
        if(i2p>i2) {
          add(Action(s1, i1, 0, s2.mid(i2, i2p-i2)));
        }
        i1+=i2m;
        i2=i2p+i2m;
      }
    }
    if(i1==s1.size()) {
      if(i2<s2.size()) {
        add(Action(s1, i1, 0, s2.mid(i2)));
      }
    } else if(i2==s2.size()) {
      if(i1<s1.size()) {
        add(Action(s1, i1, s1.size()-i1));
      }
    } else {
      ASSERT(false);
    }
  }

  void StringDiff::add(Action a)
  {
    // Check for overlapping actions
    if(!_actions.isEmpty()) {
      a.setContextStartIndex(_actions.last().endIndex());
    }
    _actions.append(a);
  }

  QString StringDiff::patch(const QString& s) const
  {
    int n=_actions.count();
    int index=0;       // Current index in s
    int matchIndex=0;
    QString final;
    for(int i=0; i<n; i++) {
      const Action& a=_actions.at(i);
      int searchIndex=index-a.beforeText().size();
      if(searchIndex<0) {
        searchIndex=0;
      }
      QString beforeText;
      int largestContext=a.beforeText().size();
      if(a.afterText().size()>largestContext) {
        largestContext=a.afterText().size();
      }
      while(largestContext>1) {
        beforeText=a.beforeText().right(largestContext);
        matchIndex=s.indexOf(beforeText, searchIndex);
        if(matchIndex!=-1) {
          break;
        } else {
          matchIndex=s.indexOf(a.afterText().left(largestContext), index);
          if(matchIndex!=-1) {
            matchIndex-=a.endIndex()-a.startIndex();
            matchIndex-=beforeText.size();
            break;
          }
        }
        largestContext--;
      }
      int nCopy=matchIndex-index+beforeText.size();
      if(nCopy>0) {
        final+=s.mid(index, nCopy);
      }
      final+=a.insertText();
      index=matchIndex+beforeText.size();
      if(a.startIndex()<a.endIndex()) {  // Searching for afterText, usefull only when removing
        int largestContext=a.afterText().size();
        while(largestContext>1) {
          matchIndex=s.lastIndexOf(a.afterText().left(largestContext));
          if(matchIndex!=-1 && matchIndex>=index) {
            index=matchIndex;
            break;
          }
          largestContext--;
        }
        if(largestContext==1) { // afterText not found, remove count characters
          index+=a.endIndex()-a.startIndex();
        }
      }
    }
    final+=s.mid(index);
    return final;
  }

  QString StringDiff::toString() const
  {
    QString str;
    int n=_actions.count();
    for(int i=0; i<n; i++) {
      str+=_actions.at(i).toString();
    }
    return str;
  }

  bool StringDiff::test(const QString& refIn, const QString& refOut,
                        const QString& in, const QString& out)
  {
    StringDiff d;
    d.setDifferences(refIn, refOut);
    App::log(1, refIn+"\n   ||\n   \\/\n"+refOut+"\n");
    QString t=d.patch(in);
    App::log(1, t+"\n");
    if(t!=out) {
      App::log("ERROR:"+t+"\n");
      App::log(d.toString());
      return false;
    } else {
      return true;
    }
  }

  void StringDiff::test()
  {
    test("Le petit cochon a construit sa maison en paille.",
         "Trois petits cochons ont construit leurs maisons en paille.",
         "Le petit chien a construit sa niche en bois.",
         "Trois petits chiens ont construit leurs niches en bois.");
    test("WA_WAU40_0001", "WAU40_A01", "WA_WAU40_0002", "WAU40_A02");
    test("WAU40_A3", "WAU40_B3", "WAU40_A2", "WAU40_B2");
    test("WA_WAU40_A3", "WAU40_A3", "WAU40_A2", "WAU40_A2");
    test("WA_WAU40_0001", "WAU40_A1", "WA_WAU40_0002", "WAU40_A2");
  }

} // namespace QGpCoreTools

