/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2016-09-02
**  Copyright: 2016-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef COLOR_H
#define COLOR_H

#include <QtCore>

#include "Translations.h"
#include "Global.h"
#include "QGpCoreToolsDLLExport.h"

#define COLOR_COMPONENT_MAX USHRT_MAX

namespace QGpCoreTools {

  class QGPCORETOOLS_EXPORT Color
  {
    TRANSLATIONS("Color")
  public:
    Color() {_rgba=0;}
    Color(int f, quint8 r, quint8 g, quint8 b, quint8 a=255) {Q_UNUSED(f);setRgba32(r, g, b, a);}
    Color(quint64 rgba) {_rgba=rgba;}
    Color(Qt::GlobalColor c);

    bool operator==(const Color& o) const {return _rgba==o._rgba;}
    bool operator!=(const Color& o) const {return _rgba!=o._rgba;}

    quint64 rgba64() const {return _rgba;}
    void setRgba64(quint64 rgba) {_rgba=rgba;}

    int red32() const {return quint8(_rgba >> (RedShift+8));}
    inline void setRed32(quint8 v);

    int green32() const {return quint8(_rgba >> (GreenShift+8));}
    inline void setGreen32(quint8 v);

    int blue32() const {return quint8(_rgba >> (BlueShift+8));}
    inline void setBlue32(quint8 v);

    int alpha32() const {return quint8(_rgba >> (AlphaShift+8));}
    inline void setAlpha32(quint8 v);

    int red64() const {return quint16(_rgba >> RedShift);}
    inline void setRed64(quint16 v);

    int green64() const {return quint16(_rgba >> GreenShift);}
    inline void setGreen64(quint16 v);

    int blue64() const {return quint16(_rgba >> BlueShift);}
    inline void setBlue64(quint16 v);

    int alpha64() const {return quint16(_rgba >> AlphaShift);}
    inline void setAlpha64(quint16 v);

    inline quint64 rgb64() const;
    inline void setRgb64(quint64 v);

    inline void setRgba32(quint8 r, quint8 g, quint8 b, quint8 a=255);
    inline void setRgba64(quint16 r, quint16 g, quint16 b, quint16 a=COLOR_COMPONENT_MAX);
    quint32 argb32() const;

    void hsv(int& h, int& s, int& v) const;
    void setHsva(int h, int s, int v, int a=255);

    void toGray();
    bool isDark() const;
    bool isNearlyWhite() const;

    QString toString() const;
    void fromString(const StringView& s, bool& ok);
  private:
    Color(quint16 r, quint16 g, quint16 b, quint16 a=COLOR_COMPONENT_MAX) {setRgba64(r, g, b, a);}

    double gammaExpansion(double csrgb);
    double gammaCompression(double clinear);
    static double unit(quint16 c);

    enum Shifts {
#if Q_BYTE_ORDER == Q_BIG_ENDIAN
        RedShift = 48,
        GreenShift = 32,
        BlueShift = 16,
        AlphaShift = 0
#else // little endian:
        RedShift = 0,
        GreenShift = 16,
        BlueShift = 32,
        AlphaShift = 48
#endif
    };
    quint64 _rgba;
  };

  inline void Color::setRgba32(quint8 r, quint8 g, quint8 b, quint8 a)
  {
    quint64 r64=r << 8;
    quint64 g64=g << 8;
    quint64 b64=b << 8;
    quint64 a64=a << 8;
    r64=r64 << RedShift;
    g64=g64 << GreenShift;
    b64=b64 << BlueShift;
    a64=a64 << AlphaShift;
    _rgba=r64 | g64 | b64 | a64;
  }

  inline void Color::setRgba64(quint16 r, quint16 g, quint16 b, quint16 a)
  {
    quint64 r64=r;
    quint64 g64=g;
    quint64 b64=b;
    quint64 a64=a;
    r64=r64 << RedShift;
    g64=g64 << GreenShift;
    b64=b64 << BlueShift;
    a64=a64 << AlphaShift;
    _rgba=r64 | g64 | b64 | a64;
  }

  inline void Color::setRed32(quint8 v)
  {
    quint64 v64=v << 8;
    quint64 mask=COLOR_COMPONENT_MAX;
    _rgba&=~(mask << RedShift);
    _rgba|=v64 << RedShift;
  }

  inline void Color::setGreen32(quint8 v)
  {
    quint64 v64=v << 8;
    quint64 mask=COLOR_COMPONENT_MAX;
    _rgba&=~(mask << GreenShift);
    _rgba|=v64 << GreenShift;
  }

  inline void Color::setBlue32(quint8 v)
  {
    quint64 v64=v << 8;
    quint64 mask=COLOR_COMPONENT_MAX;
    _rgba&=~(mask << BlueShift);
    _rgba|=v64 << BlueShift;
  }

  inline void Color::setAlpha32(quint8 v)
  {
    quint64 v64=v << 8;
    quint64 mask=COLOR_COMPONENT_MAX;
    _rgba&=~(mask << AlphaShift);
    _rgba|=v64 << AlphaShift;
  }

  inline void Color::setRed64(quint16 v)
  {
    quint64 v64=v;
    quint64 mask=COLOR_COMPONENT_MAX;
    _rgba&=~(mask << RedShift);
    _rgba|=v64 << RedShift;
  }

  inline void Color::setGreen64(quint16 v)
  {
    quint64 v64=v;
    quint64 mask=COLOR_COMPONENT_MAX;
    _rgba&=~(mask << GreenShift);
    _rgba|=v64 << GreenShift;
  }

  inline void Color::setBlue64(quint16 v)
  {
    quint64 v64=v;
    quint64 mask=COLOR_COMPONENT_MAX;
    _rgba&=~(mask << BlueShift);
    _rgba|=v64 << BlueShift;
  }

  inline void Color::setAlpha64(quint16 v)
  {
    quint64 v64=v;
    quint64 mask=COLOR_COMPONENT_MAX;
    _rgba&=~(mask << AlphaShift);
    _rgba|=v64 << AlphaShift;
  }

  inline quint64 Color::rgb64() const
  {
    quint64 mask=COLOR_COMPONENT_MAX;
    mask=~(mask << AlphaShift);
    return _rgba & mask;
  }

  inline void Color::setRgb64(quint64 v)
  {
    quint64 v64=v;
    quint64 mask=COLOR_COMPONENT_MAX;
    _rgba&=mask << AlphaShift;
    _rgba|=v64;
  }

} // namespace QGpCoreTools

#endif // COLOR_H

