/***************************************************************************
**
**  This file is part of QGpCoreStat.
**
**  QGpCoreStat is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreStat is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-12-08
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef MULTIVARIATEHISTOGRAM_H
#define MULTIVARIATEHISTOGRAM_H

#include <QGpCoreMath.h>

#include "QGpCoreStatDLLExport.h"

namespace QGpCoreStat {

  class GaussianMixtureDistribution;

  class QGPCORESTAT_EXPORT MultivariateHistogram : private IncreaseStorage
  {
  public:
    MultivariateHistogram(int dimensionCount);
    MultivariateHistogram(const MultivariateHistogram& o)=delete;
    ~MultivariateHistogram();

    void setSampling(int dim, const SamplingParameters& s, bool periodic=false);
    const SamplingParameters& sampling(int dim) const {return _samplings[dim];}

    int dimensionCount() const {return _bucket.count();}
    int bucketCount() const {return size();}
    int sampleCount() const;

    void addSample(const Vector<double>& x);
    void removeSmallBucketValues(double minimum);
    void smooth();
    void normalize();
    void effectiveLimits(Vector<double>& min, Vector<double>& max) const;
    int indexAfter(const Vector<int>& bucket, bool& matched) const;

    bool isEmpty() const {return size()==0;}

    inline void bucketPosition(int index, Vector<double>& pos) const;
    inline const Vector<int>& bucketAt(int index) const;
    inline const int& bucketValueAt(int index) const;
    inline int& bucketValueAt(int index);

    inline double normalizedBucketValueAt(int index) const;
    double misfit(const GaussianMixtureDistribution& f) const;
    MultivariateStatistics * statistics() const;

    void printDebug();
  private:
    void reallocate();

    SamplingParameters * _samplings;
    bool * _periodic;
    PrivateVector<int> _newBucket;
    mutable Vector<int> _bucket;
    int _bucketCount2;
    int * _data;
    double _normalizationFactor;
  };

  inline void MultivariateHistogram::bucketPosition(int index, Vector<double>& pos) const
  {
    const Vector<int>& bucket=bucketAt(index);
    for(int i=bucket.count()-1; i>=0; i--) {
      pos[i]=_samplings[i].value(bucket[i]);
    }
  }

  inline const Vector<int>& MultivariateHistogram::bucketAt(int index) const
  {
    _bucket.setValues(_data+(_bucket.count()+1)*index);
    return _bucket;
  }

  inline const int& MultivariateHistogram::bucketValueAt(int index) const
  {
    return _data[(_bucket.count()+1)*index+_bucket.count()];
  }

  inline int& MultivariateHistogram::bucketValueAt(int index)
  {
    return _data[(_bucket.count()+1)*index+_bucket.count()];
  }

  inline double MultivariateHistogram::normalizedBucketValueAt(int index) const
  {
    return static_cast<double>(bucketValueAt(index))*_normalizationFactor;
  }

} // namespace QGpCoreStat

#endif // MULTIVARIATEHISTOGRAM_H

