/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-03-24
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef POLYNOMIALFRACTION_H
#define POLYNOMIALFRACTION_H

#include "QGpCoreMathDLLExport.h"
#include "Polynomial.h"

namespace QGpCoreMath {

  class QGPCOREMATH_EXPORT PolynomialFraction
  {
  public:
    PolynomialFraction(int numeratorDegree, int denominatorDegree)
      : _numerator(numeratorDegree), _denominator(denominatorDegree) {}
    PolynomialFraction(const Polynomial& numerator,
                       const Polynomial& denominator)
      : _numerator(numerator), _denominator(denominator) {}
    PolynomialFraction(const Polynomial& numerator)
      : _numerator(numerator), _denominator(0) {_denominator.setCoefficient(0, 1.0);}

    Polynomial& numerator() {return _numerator;}
    const Polynomial& numerator() const {return _numerator;}

    Polynomial& denominator() {return _denominator;}
    const Polynomial& denominator() const {return _denominator;}

    inline void operator*=(const Polynomial& o);
    inline void operator/=(const Polynomial& o);
    inline void operator*=(const PolynomialFraction& o);
    inline void operator/=(const PolynomialFraction& o);
    inline void operator+=(const PolynomialFraction& o);
    inline void operator-=(const PolynomialFraction& o);
    inline PolynomialFraction operator+(const PolynomialFraction& o) const;
    inline PolynomialFraction operator-(const PolynomialFraction& o) const;
    inline PolynomialFraction operator*(const PolynomialFraction& o) const;
    inline PolynomialFraction operator/(const PolynomialFraction& o) const;
    inline PolynomialFraction operator+(const Polynomial& o) const;
    inline PolynomialFraction operator-(const Polynomial& o) const;
    inline PolynomialFraction operator*(const Polynomial& o) const;
    inline PolynomialFraction operator/(const Polynomial& o) const;
    inline void operator+=(double f);
    inline void operator-=(double f);
    void operator*=(double f) {_numerator*=f;}
    void operator/=(double f) {_numerator/=f;}

    void inverse();
    void derivate();

    QString toString() const;
    double value(double x) const {return _numerator.value(x)/_denominator.value(x);}
  private:
    Polynomial _numerator, _denominator;
  };

  inline void PolynomialFraction::operator*=(const Polynomial& o)
  {
    _numerator=_numerator*o;
  }

  inline void PolynomialFraction::operator/=(const Polynomial& o)
  {
    _denominator=_denominator*o;
  }

  inline void PolynomialFraction::operator*=(const PolynomialFraction& o)
  {
    _numerator=_numerator*o._numerator;
    _denominator=_denominator*o._denominator;
  }

  inline void PolynomialFraction::operator/=(const PolynomialFraction& o)
  {
    _numerator=_numerator*o._denominator;
    _denominator=_denominator*o._numerator;
  }

  inline void PolynomialFraction::operator+=(const PolynomialFraction& o)
  {
    bool ok;
    double f=_denominator.multiple(o._denominator, ok);
    if(ok) {
      _numerator+=o._numerator/f;
    } else {
      _numerator=_numerator*o._denominator;
      _numerator+=o._numerator*_denominator;
      _denominator=_denominator*o._denominator;
    }
  }

  inline void PolynomialFraction::operator-=(const PolynomialFraction& o)
  {
    bool ok;
    double f=_denominator.multiple(o._denominator, ok);
    if(ok) {
      _numerator-=o._numerator/f;
    } else {
      _numerator=_numerator*o._denominator;
      _numerator-=o._numerator*_denominator;
      _denominator=_denominator*o._denominator;
    }
  }

  inline void PolynomialFraction::operator+=(double f)
  {
    _numerator+=_denominator*f;
  }

  inline void PolynomialFraction::operator-=(double f)
  {
    _numerator-=_denominator*f;
  }

  inline PolynomialFraction PolynomialFraction::operator+(const PolynomialFraction& o) const
  {
    bool ok;
    double f=_denominator.multiple(o._denominator, ok);
    if(ok) {
      return PolynomialFraction(_numerator+o._numerator/f,
                                _denominator);
    } else {
      return PolynomialFraction(_numerator*o._denominator+o._numerator*_denominator,
                                _denominator*o._denominator);
    }
  }

  inline PolynomialFraction PolynomialFraction::operator-(const PolynomialFraction& o) const
  {
    bool ok;
    double f=_denominator.multiple(o._denominator, ok);
    if(ok) {
      return PolynomialFraction(_numerator-o._numerator/f,
                                _denominator);
    } else {
      return PolynomialFraction(_numerator*o._denominator-o._numerator*_denominator,
                                _denominator*o._denominator);
    }
  }

  inline PolynomialFraction PolynomialFraction::operator*(const PolynomialFraction& o) const
  {
    return PolynomialFraction(_numerator*o._numerator,
                              _denominator*o._denominator);
  }

  inline PolynomialFraction PolynomialFraction::operator/(const PolynomialFraction& o) const
  {
    return PolynomialFraction(_numerator*o._denominator,
                              _denominator*o._numerator);
  }

  inline PolynomialFraction PolynomialFraction::operator+(const Polynomial& o) const
  {
    return PolynomialFraction(_numerator+o*_denominator, _denominator);
  }

  inline PolynomialFraction PolynomialFraction::operator-(const Polynomial& o) const
  {
    return PolynomialFraction(_numerator-o*_denominator, _denominator);
  }

  inline PolynomialFraction PolynomialFraction::operator*(const Polynomial& o) const
  {
    return PolynomialFraction(_numerator*o, _denominator);
  }

  inline PolynomialFraction PolynomialFraction::operator/(const Polynomial& o) const
  {
    return PolynomialFraction(_numerator, _denominator*o);
  }

} // namespace QGpCoreMath

#endif // POLYNOMIALFRACTION_H

