/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-08-19
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef OPTIMIZATIONBFGS_H
#define OPTIMIZATIONBFGS_H

#include "FunctionSearch.h"
#include "Vector.h"
#include "DoubleMatrix.h"
#include "QGpCoreMathDLLExport.h"

namespace QGpCoreMath {

  class QGPCOREMATH_EXPORT OptimizationBFGS : public FunctionSearch
  {
  public:
    OptimizationBFGS(int dimensionCount);
    ~OptimizationBFGS();

    void setPrecision(const Vector<double>& p) {_precision.copyValues(p);}

    void reset(const Vector<double>& p);
    bool minimize(const Vector<double>& maxShift, int maxIterationCount);

    double value() const {return _phi.functionProperties()->value;}
    Vector<double>& position() {return _phi.x0();}
    int iterationCount() const {return _iterationCount;}
  private:
    class Phi;

    class Step
    {
    public:
      Step() {_alpha=0.0; _dphi=0.0; _f=nullptr;}

      void setAlpha(double a) {_alpha=a;}
      double alpha() const {return _alpha;}

      void setValue(double v) {_f->value=v;}
      double value() const {return _f->value;}

      void setFunctionProperties(AbstractFunction::Properties * f) {_f=f;}
      AbstractFunction::Properties *& functionProperties() {return _f;}

      void setDerivative(const Vector<double>& searchDirection);
      double derivative() const {return _dphi;}
    private:
      double _alpha, _dphi;
      AbstractFunction::Properties * _f;
    };

    class Phi
    {
    public:
      inline Phi(int dimensionCount);

      void setFunctionProperties(AbstractFunction::Properties * f) {_f0=f;}
      void setFunctionProperties(Step& s) {qSwap(_f0, s.functionProperties());}
      AbstractFunction::Properties * functionProperties() {return _f0;}
      const AbstractFunction::Properties * functionProperties() const {return _f0;}

      inline void setSearchDirection(const DoubleMatrix& h);
      const Vector<double>& searchDirection() const {return _p;}

      void move(const Vector<double>& step) {_x0+=step;}
      Vector<double>& x0() {return _x0;}
      inline void setArgument(double alpha, Vector<double>& arg) const;

      inline bool isDecreaseCondition(const Step& s) const;
      inline bool isIncreaseCondition(const Step& s) const;
      inline bool isCurvatureCondition(const Step& s) const;
    private:
      PrivateVector<double> _x0;
      AbstractFunction::Properties  * _f0;
      PrivateVector<double> _p;
      double _c1k0, _c2k0;
    };

    inline static double quadratic(double x1, double y1, double dy1,
                                   double x2, double y2);
    inline double bisection(double x1, double x2);

    OptimizationBFGS::Step * bracketing();
    OptimizationBFGS::Step * zoom(Step *& low, Step *& high);
    void debugAlpha(double min, double max);
    void debugFunction(double range);

    // Parameter
    PrivateVector<double> _precision;
    // Working members
    Step * _steps[3];
    Phi _phi;
    AbstractFunction::Properties * _fValues[4];
    PrivateVector<double> _absStep;
    PrivateVector<double> _absShift;
    PrivateVector<double> _sk;
    PrivateVector<double> _yk;
    PrivateVector<double> _x;
    PrivateVector<double> _x0;
    DoubleMatrix _hk;
    DoubleMatrix _tk;
    DoubleMatrix _skskt;
    double _dalpha;
    int _iterationCount;
  };

} // namespace QGpCoreMath

#endif // OPTIMIZATIONBFGS_H

