/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-11-25
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "FrequencyCurveProxy.h"

namespace QGpCoreMath {

  /*!
    \class FrequencyCurveProxy FrequencyCurveProxy.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  QString FrequencyCurveProxy::columnName(int col) const
  {
    TRACE;
    switch(col) {
    case 0: return tr("Frequency");
    case 1: return tr("Period");
    case 2: return tr("Y");
    case 3: return tr("Stddev");
    case 4: return tr("Weight");
    default: return QString();
    }
  }

  QString FrequencyCurveProxy::columnUnit(int col) const
  {
    TRACE;
    switch(col) {
    case 0: return tr("Hz");
    case 1: return tr("s");
    default: return QString();
    }
  }

  VectorList<int> FrequencyCurveProxy::savedColumns() const
  {
    TRACE;
    VectorList<int> l;
    l << 0 << 2 << 3 << 4;
    return l;
  }

  QVariant FrequencyCurveProxy::columnValue(int sample, int col) const
  {
    TRACE;
    const RealStatisticalPoint& p=curve().constAt(sample);
    switch(col) {
    case 0: return p.x();
    case 1: return 1.0/p.x();
    case 2: return p.mean();
    case 3: return p.stddev();
    case 4: return p.weight();
    default: return QVariant();
    }
  }

  void FrequencyCurveProxy::setColumnValue(int sample, int col, const QVariant & value)
  {
    TRACE;
    RealStatisticalPoint& p=curve().constXAt(sample);
    switch(col) {
    case 0: curve().setX(sample, value.toDouble()); break;
    case 1: curve().setX(sample, 1.0/value.toDouble()); break;
    case 2: p.setMean(value.toDouble()); break;
    case 3: p.setStddev(value.toDouble()); break;
    case 4: p.setWeight(value.toDouble()); break;
    default: break;
    }
  }

  QStringList FrequencyCurveProxy::columnFileTypes() const
  {
    TRACE;
    static QStringList types;
    if(types.isEmpty()) {
      types << tr("Frequency (Hz)")          // 1
            << tr("Period (s)")              // 2
            << tr("Average")                 // 3
            << tr("Stddev")                  // 4
            << tr("Weight")                  // 5
            << tr("Valid");                  // 6
    }
    return types;
  }

  VectorList<int> FrequencyCurveProxy::defaultColumnFileTypes() const
  {
    TRACE;
    VectorList<int> types;
    types << 1 << 3 << 4 << 5 << 6;
    return types;
  }

  bool FrequencyCurveProxy::parse(ColumnTextIterator& it)
  {
    TRACE;
    const ColumnTextParser * parser=it.parser();
    int nColumns=parser->columnCount();
    Curve<RealStatisticalPoint>& c=curve();
    c.clear();
    bool ok=true;
    while(!it.atEnd() && c.isEmpty()) {
      while(!it.atSectionEnd()) {
        RealStatisticalPoint p;
        for(int iCol=0;iCol<nColumns;iCol++) {
          switch(parser->type(iCol)) {
          case 1:
            p.setX(parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
          case 2:
            p.setX(1.0/parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
          case 3:
            p.setMean(parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
          case 4:
            p.setStddev(parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
          case 5:
            p.setWeight(parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
          case 6:
            p.setValid(parser->text(it.currentRow(), iCol).toInt(&ok)==1);
            break;
          default:
            break;
          }
          if(!ok) {
            parser->errorParsingColumn(iCol, it.currentRow());
            return false;
          }
        }
        c.append(p);
        it.nextRow();
      }
    }
    return true;
  }

} // namespace QGpCoreMath

