/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <math.h>

#include "DirectionalSearch.h"

namespace QGpCoreMath {

/*!
  \class DirectionalSearch DirectionalSearch.h
  \brief Search a grid for maxima (global or local) along one direction only

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
DirectionalSearch::DirectionalSearch()
  : FunctionSearch(2)
{
  TRACE;
  _precision[0]=0.001;
  _cx=1.0;
  _sy=0.0;
}

/*!
  Description of destructor still missing
*/
DirectionalSearch::~DirectionalSearch()
{
  TRACE;
}

/*!
  Set current azimuth of search
*/
void DirectionalSearch::setAzimuth(double azimuth)
{
  TRACE;
  _cx=::cos(azimuth);
  _sy=::sin(azimuth);
}

/*!
  Set gridding along radius
*/
void DirectionalSearch::setGrid(double min, double max, double step)
{
  TRACE;
  ASSERT(max-min>step); // At least 1 point
  _step=step;
  _min=min;
  _count=qFloor((max-min)/step);
}

inline void DirectionalSearch::setCurrentPosition(int i)
{
  double r=_min+_step*i;
  _currentPosition[0]=_cx*r;
  _currentPosition[1]=_sy*r;
}

void DirectionalSearch::globalMax(double absThres)
{
  TRACE;
  clearMaxima();
  double val, maxVal=-std::numeric_limits<double>::infinity();
  int maxk=0;
  for(int k=0; k<_count; k++) {
    setCurrentPosition(k);
    val=value(_currentPosition);
    if(val>maxVal) {
      maxVal=val;
      maxk=k;
    }
  }
  setCurrentPosition(maxk);
  addMaximum(maxVal);
  FunctionSearch::globalMax(absThres);
}

/*!
  Description still missing
*/
void DirectionalSearch::localMax(int nMax, double absThres, double relThres)
{
  TRACE;
  clearMaxima();
  double previous, current, next;
  setCurrentPosition(0);
  previous=value(_currentPosition);
  setCurrentPosition(1);
  current=value(_currentPosition);
  for(int k=2; k<_count; k++) {
    setCurrentPosition(k);
    next=value(_currentPosition);
    if(current>next &&
      current>previous) {
      setCurrentPosition(k-1);
      addMaximum(current);
    }
    previous=current;
    current=next;
  }
  FunctionSearch::localMax(nMax, absThres, relThres);
}

/*!
  Refines the maximum of the grid in an area defined by current position +/- one step.
  Returns value of the refined maximum
*/
double DirectionalSearch::refineMax(double val)
{
  TRACE;
  double absPrecision;
  if(_relative[0]) {
    absPrecision=fabs(_currentPosition[0]*_precision[0]);
    if(absPrecision==0.0) { // Make sure that precision is not null
      absPrecision=fabs(_precision[0]*(_currentPosition[0]+_step*0.5));
    }
  } else {
    absPrecision=_precision[0];
  }

  double step=_step;
  while(step>absPrecision) {
    step*=0.5;
    _testPosition[0]=_testPosition[0]-step;
    testForMax(val);
    _testPosition[0]=_testPosition[0]+step;
    testForMax(val);
  }
  return val;
}

} // namespace QGpCoreMath
