/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-05-24
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "ColorMap.h"

namespace QGpCoreMath {

  /*!
    \class ColorMap ColorMap.h
    \brief A ColorMap links a color palette with a vector of values

    ColorMap are defined by a QColor vector of n elements and a vector
    of double of n-1 elements. Each value at i corresponds to the interface between
    the two colors at positions i and i+1.

    Article to read
    https://www.nature.com/articles/s41467-020-19160-7
  */

  const QString ColorMap::xmlColorMapTag="ColorMap";

  bool ColorMap::xml_isValidTagName(const QString& t) const
  {
    return t=="ColorPalette"; // Compatibility with old release
  }

  ColorPalette ColorMap::palette() const
  {
    return _d->palette();
  }

  void ColorMap::setPalette(const ColorPalette& pal)
  {
    _d->setPalette(pal);
  }

  void ColorMap::generate(int imin, int imax, const QString& model,
                          ColorPalette::Options options, quint16 transparency)
  {
    _d->generate(imin, imax, model, options, transparency);
  }

  void ColorMap::toGray()
  {
    _d->toGray();
  }

  void ColorMap::rgbInterpole(int imin, int imax)
  {
    _d->rgbInterpole(imin, imax);
  }

  void ColorMap::hsvInterpole(int imin, int imax)
  {
    _d->hsvInterpole(imin, imax);
  }

  void ColorMap::setWhiteTransparent(bool b)
  {
    _d->setWhiteTransparent(b);
  }

  void ColorMap::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
  {
    TRACE;
    Q_UNUSED(context)
    writeProperty(s, "nColors", count());
    QString tmp;
    tmp+=s.indent();
    tmp+="<colorMap>\n";
    int n=count()-1;
    for(int i=0; i<n; i++) {
      tmp+=s.indent();
      tmp+=color(i).toString();
      tmp+=" ";
      tmp+=QString::number(upperValue(i));
      tmp+="\n";
    }
    tmp+=s.indent();
    tmp+=color(n).toString();
    tmp+="\n";
    tmp+=s.indent();
    tmp+="</colorMap>\n";
    s << tmp;
  }

  XMLMember ColorMap::xml_member(XML_MEMBER_ARGS)
  {
    TRACE;
    Q_UNUSED(attributes)
    Q_UNUSED(context)
    if(tag=="nColors") return XMLMember(0);
    else if(tag=="colorMap") return XMLMember(3);
    else if(tag=="numberType") return XMLMember(1);       // Compatibility
    else if(tag=="numberPrecision") return XMLMember(2);  // Compatibility
    else if(tag=="palette") return XMLMember(3);          // Compatibility
    else if(tag=="numType") return XMLMember(1);          // Compatibility
    else if(tag=="numPrec") return XMLMember(2);          // Compatibility
    else return XMLMember(XMLMember::Unknown);
  }

  bool ColorMap::xml_setProperty(XML_SETPROPERTY_ARGS)
  {
    TRACE;
    Q_UNUSED(tag)
    Q_UNUSED(attributes)
    Q_UNUSED(context)
    bool ok=true;
    switch(memberID) {
    case 0: {
        int n=content.toInt(ok);
        if(ok && n>0) {
          resize(n);
          return true;
        } else return false;
      }
    case 1:
    case 2:
      return true;
    case 3: {
        if(count()==0) return false;
        const QChar * ptr=nullptr;
        content.nextLine(ptr);
        StringSection f;
        int i;
        Color c;
        double val;
        int n=count()-1;
        for(i=0; i<n; i++) {
          f=content.nextField(ptr);
          c.fromString(f.toStringBuffer(), ok);
          if(f.isValid() && ok) {
            setColor(i, c);
          } else {
            break;
          }
          f=content.nextField(ptr);
          val=f.toDouble(ok);
          if(f.isValid() && ok) {
            setUpperValue(i, val);
          } else {
            break;
          }
          content.nextLine(ptr);
        }
        if(ok) {
          // Last color without value
          f=content.nextField(ptr);
          c.fromString(f.toStringView(), ok);
          if(f.isValid() && ok) {
            setColor(i, c);
          }
          return true;
        } else {
          App::log(tr("Error parsing color %1 in line '%2'\n")
                   .arg(i)
                   .arg(content.toStringView()));
          return false;
        }
      }
    default:
      break;
    }
    return false;
  }

} // namespace QGpCoreMath
