/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef ABSTRACTFUNCTION_H
#define ABSTRACTFUNCTION_H

#include <QtCore>

#include "QGpCoreMathDLLExport.h"
#include "FunctionSearchMaximum.h"
#include "PrivateVector.h"

namespace QGpCoreMath {

  class FunctionSearchMaximum;

  class QGPCOREMATH_EXPORT AbstractFunction
  {
  public:
    AbstractFunction() {}
    virtual ~AbstractFunction() {}

    class Properties
    {
    public:
      Properties(int dimensionCount) : gradient(dimensionCount, 0.0) {value=0.0;}

      double value;
      PrivateVector<double> gradient;
    };

    virtual double value(const Vector<int>& index) const;
    virtual double value(Vector<double>& x) const;
    virtual double valueFixed(const Vector<double>& x) const;
    virtual void gradient(const Vector<double>& x, Vector<double>& grad) const;
    virtual double admissibleStep(const Vector<double>& x, const Vector<double>& step) const;

    virtual double concavity(const Vector<double>& x, int axis) const {Q_UNUSED(x); Q_UNUSED(axis); return 0.0;}
    virtual void stepDirection(const Vector<double>& x, Vector<double>& dir) const;

    double maximumValue(Vector<double>& x, int axis) const;
    inline double gradientLength2D(const Vector<double>& x, int axis1, int axis2) const;
    inline double gradientDirection2D(const Vector<double>& x, int axis1, int axis2) const;
    inline double stepLength2D(const Vector<double>& x, int axis1, int axis2) const;
    inline double stepDirection2D(const Vector<double>& x, int axis1, int axis2) const;
  };

  inline double AbstractFunction::gradientLength2D(const Vector<double>& x, int axis1, int axis2) const
  {
    PrivateVector<double> g(x.count(), 0.0);
    gradient(x, g);
    return g.length(axis1, axis2);
  }

  inline double AbstractFunction::gradientDirection2D(const Vector<double>& x, int axis1, int axis2) const
  {
    PrivateVector<double> g(x.count(), 0.0);
    gradient(x, g);
    return g.azimuth(axis1, axis2);
  }

  inline double AbstractFunction::stepLength2D(const Vector<double>& x, int axis1, int axis2) const
  {
    PrivateVector<double> dir(x.count(), 0.0);
    stepDirection(x, dir);
    return dir.length(axis1, axis2);
  }

  inline double AbstractFunction::stepDirection2D(const Vector<double>& x, int axis1, int axis2) const
  {
    PrivateVector<double> dir(x.count(), 0.0);
    stepDirection(x, dir);
    return dir.azimuth(axis1, axis2);
  }

} // namespace QGpCoreMath

#endif // ABSTRACTFUNCTION2.H
