/***************************************************************************
**
**  This file is part of HVGui.
**
**  HVGui is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  HVGui is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2020-06-22
**  Copyright: 2020
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "AveragePeakModel.h"

namespace HVGui {

  /*!
    \class AveragePeakModel AveragePeakModel.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  AveragePeakModel::AveragePeakModel(const QList<CurveResults *> * results,
                                     const VectorList<RealStatisticalValue>& peaks,
                                     QObject * parent)
    : AbstractPeakModel(parent)
  {
    TRACE;
    _res=results;
    _peaks=peaks;
  }

  /*!
    Description of destructor still missing
  */
  AveragePeakModel::~AveragePeakModel()
  {
    TRACE;
  }

  int AveragePeakModel::rowCount(const QModelIndex & parent) const
  {
    TRACE;
    if(!parent.isValid()) {
      return _peaks.count();
    } else {
      return 0;
    }
  }

  int AveragePeakModel::columnCount(const QModelIndex & parent) const
  {
    TRACE;
    if(!parent.isValid()) {
      return 3;
    } else {
      return 0;
    }
  }

  QVariant AveragePeakModel::data(const QModelIndex &index, int role) const
  {
    TRACE;
    const RealStatisticalValue& p=_peaks.at(index.row());
    switch (role) {
    case Qt::DisplayRole:
      switch(index.column()) {
      case 0:
        return p.mean();
      case 1:
        return p.mean()/p.stddev();
      case 2:
        return p.mean()*p.stddev();
      default: break;
      }
      break;
    default:
      break;
    }
    return QVariant();
  }

  QVariant AveragePeakModel::headerData(int section, Qt::Orientation orientation, int role) const
  {
    TRACE;
    if(role!=Qt::DisplayRole) return QVariant();
    if(orientation==Qt::Horizontal) {
      switch(section) {
      case 0: return tr("Frequency (Hz)");
      case 1: return tr("f/stddev (Hz)");
      case 2: return tr("f*stddev (Hz)");
      default: break;
      }
    } else {
      return section+1;
    }
    return QVariant();
  }

  RealStatisticalValue AveragePeakModel::peak(double min, double max)
  {
    Statistics s;
    for(QList<CurveResults *>::const_iterator it=_res->begin(); it!=_res->end(); it++) {
      CurveResults * res=*it;
      double f=res->peak(min, max, CurveResults::PeakFrequency);
      if(f>0.0) {
        s.addLog(f);
      }
    }
    if(s.count()==0) {
      App::log(tr("No peak found in the range from %1 to %2 Hz.\n")
               .arg(min).arg(max));
    }
    return RealStatisticalValue(s.meanLog(),
                                s.stddevLog(),
                                s.count(),
                                s.count()>0);
  }

  void AveragePeakModel::add(double min, double max)
  {
    TRACE;
    int n=_peaks.count();
    RealStatisticalValue p=peak(min, max);
    if(p.isValid()) {
      beginInsertRows(QModelIndex(), n, n);
      _peaks.append(p);
      endInsertRows();
    }
  }

  void AveragePeakModel::remove(int index)
  {
    TRACE;
    beginRemoveRows(QModelIndex(), index, index);
    _peaks.removeAt(index);
    endRemoveRows();
  }

  void AveragePeakModel::adjust(int index, double min, double max)
  {
    TRACE;
    RealStatisticalValue p=peak(min, max);
    if(p.isValid()) {
      _peaks[index]=p;
    } else {
      beginRemoveRows(QModelIndex(), index, index);
      _peaks.removeAt(index);
      endRemoveRows();
    }
  }

  void AveragePeakModel::save(QTextStream& s, int index)
  {
    const RealStatisticalValue& p=_peaks.at(index);
    s << "Average\t" << p.mean()
      << "\t" << p.mean()/p.stddev()
      << "\t" << p.mean()*p.stddev() << "\n";
  }

} // namespace HVGui

