/***************************************************************************
**
**  This file is part of GeopsyPySciFigs.
**
**  GeopsyPySciFigs is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyPySciFigs is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-11-24
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifdef HAS_PYTHON
#include "PythonNumPy.h"

#include "PythonClassFactory.h"
#include "GraphicalUserInterface.h"
#include "MethodArgument.h"
#include "MethodSort.h"

/*!
  \class PythonClassFactory PythonClassFactory.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

PythonClassFactory * PythonClassFactory::_self=nullptr;

struct QtObjectData {
  PyObject_HEAD
  int objectId;
};

static PyTypeObject typeTemplate={
    PyVarObject_HEAD_INIT(NULL, 0)
    0,                       // tp_name
    sizeof(QtObjectData),    // tp_basicsize
    0,                       // tp_itemsize
    0,                       // tp_dealloc
    0,                       // tp_print or tp_vectorcall_offset (cpython/object.h)
    0,                       // tp_getattr
    0,                       // tp_setattr
    0,                       // tp_reserved
    0,                       // tp_repr
    0,                       // tp_as_number
    0,                       // tp_as_sequence
    0,                       // tp_as_mapping
    0,                       // tp_hash
    0,                       // tp_call
    0,                       // tp_str
    0,                       // tp_getattro
    0,                       // tp_setattro
    0,                       // tp_as_buffer
    Py_TPFLAGS_DEFAULT,      // tp_flags
    0,                       // tp_doc
    0,                       // tp_traverse
    0,                       // tp_clear
    0,                       // tp_richcompare
    0,                       // tp_weaklistoffset
    0,                       // tp_iter
    0,                       // tp_iternext
    0,                       // tp_methods
    0,                       // tp_members
    0,                       // tp_getset
    0,                       // tp_base
    0,                       // tp_dict
    0,                       // tp_descr_get
    0,                       // tp_descr_set
    0,                       // tp_dictoffset
    0,                       // tp_init
    0,                       // tp_alloc
    PyType_GenericNew,       // tp_new
    0,                       // tp_free Low-level free-memory routine
    0,                       // tp_is_gc For PyObject_IS_GC
    0,                       // tp_bases
    0,                       // tp_mro method resolution order
    0,                       // tp_cache
    0,                       // tp_subclasses
    0,                       // tp_weaklist
    0,                       // tp_del;
    0,                       // tp_version_tag
    0,                       // tp_finalize
    0                        // tp_vectorcall
};

#define VALUE_FUNCTION(h, t, u) \
{ \
      int index=100*h+10*t+u; \
      _mapValues[index]=(PyCFunction) value_##h##t##u; \
      _mapSetValues[index]=(PyCFunction) setValue_##h##t##u; \
}

#define METHOD_FUNCTION(h, t, u) \
{ \
      int index=100*h+10*t+u; \
      _mapMethods[index]=(PyCFunction) method_##h##t##u; \
}

/*!
  Description of constructor still missing
*/
PythonClassFactory::PythonClassFactory()
{
  TRACE;
  ASSERT(!_self);
  _self=this;

  _mapValues=new PyCFunction[MULITPLE_FUNCTION_COUNT];
  _mapSetValues=new PyCFunction[MULITPLE_FUNCTION_COUNT];
  _mapMethods=new PyCFunction[MULITPLE_FUNCTION_COUNT];
  MULTIPLE_FUNCTION(VALUE_FUNCTION)
  MULTIPLE_FUNCTION(METHOD_FUNCTION)
}

/*!
  Description of destructor still missing
*/
PythonClassFactory::~PythonClassFactory()
{
  TRACE;
  QMap<QString, TypeInfo>::iterator it;
  for(it=_types.begin(); it!=_types.end(); it++) {
    delete [] it.value().pyObject->tp_methods;
    delete it.value().pyObject;
  }
}

void PythonClassFactory::cleanFunctionMaps()
{
  delete [] _mapValues;
  _mapValues=nullptr;
  delete [] _mapSetValues;
  _mapSetValues=nullptr;
  delete [] _mapMethods;
  _mapMethods=nullptr;
}

bool PythonClassFactory::add(XMLClassFactory * factory)
{
  QList<int> ids=factory->registeredIds();
  for(int i=0; i<ids.size(); i++) {
    XMLClassCreator * c=factory->creator(ids.at(i));
    if(c) {
      const QMetaObject * mObj=c->metaObject();
      if(mObj) {
        if(!add(mObj)) {
          return false;
        }
      }
    }
  }
  return true;
}

bool PythonClassFactory::add(const QMetaObject * object)
{
  TypeInfo type;

  // Get method and property offset for first class above QWidget
  const QMetaObject * superObject=object->superClass();
  while(superObject &&
        strcmp(superObject->className(),"QWidget")!=0 &&
        strcmp(superObject->className(),"QObject")!=0) {
    superObject=superObject->superClass();
  }
  if(superObject) {
    type.propertyOffset=superObject->propertyCount();
    type.methodOffset=superObject->methodCount();
  } else {
    type.propertyOffset=0;
    type.methodOffset=0;
  }
  type.qtObject=object;
  type.pyObject=new PyTypeObject(typeTemplate);
  type.pyObject->tp_name=object->className();
  type.pyObject->tp_doc=PyDoc_STR("GeopsyPySciFigs object");

  _types.insert(object->className(), type);
  //printf("Add class %s\n", object->className());
  return addMethods(_types.find(object->className()).value());
}

bool PythonClassFactory::addMethods(TypeInfo& type)
{
  // Counts properties
  int nProperties=type.qtObject->propertyCount()-type.propertyOffset;
  if(nProperties>MULITPLE_FUNCTION_COUNT) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("Number of properties in class %1 exceed %3 (%2)")
                                                            .arg(type.qtObject->className())
                                                            .arg(nProperties)
                                                            .arg(MULITPLE_FUNCTION_COUNT));
    return false;
  }
  int nAcceptedProperties=0;
  for(int i=0; i<nProperties; i++) {
    QMetaProperty p=type.qtObject->property(type.propertyOffset+i);
    if(p.isStored()) {
      nAcceptedProperties++;
    }
  }

  // Counts compatible methods
  // Non slot/signal function must be Q_INVOKABLE
  // Get index of first and last usefull method
  // Python does not support overloaded function
  // Virtual functions with the same arguments are referenced only one
  int nMethods=type.qtObject->methodCount()-type.methodOffset;
  QMap<MethodSort, int> methodSort;
  for(int i=0; i<nMethods; i++) {
    QMetaMethod m=type.qtObject->method(type.methodOffset+i);
    if(isAcceptableMethod(m)) {
      methodSort.insert(m, i);
    }
  }
  QMap<MethodSort, int>::iterator it;
  struct OverLoadIndex {
    int count;
    int index;
  };
  int firstMethodIndex=nMethods-1;
  int lastMethodIndex=0;
  QMap<QString, OverLoadIndex> methodOverloads;
  for(it=methodSort.begin(); it!=methodSort.end(); it++) {
    int index=it.value();
    QMetaMethod m=type.qtObject->method(type.methodOffset+index);
    if(methodOverloads.contains(m.name())) {
      methodOverloads[m.name()].count++;
    } else {
      methodOverloads.insert(m.name(), {1, 0});
    }
    if(index>lastMethodIndex) {
      lastMethodIndex=index;
    }
    if(index<firstMethodIndex) {
      firstMethodIndex=index;
    }
  }
  int nAcceptedMethods=methodSort.count();
  nMethods=lastMethodIndex-firstMethodIndex+1;
  type.methodOffset+=firstMethodIndex;
  if(nMethods>MULITPLE_FUNCTION_COUNT) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("Number of methods in class %1 exceed %3 (%2)")
                                                            .arg(type.qtObject->className())
                                                            .arg(nMethods)
                                                            .arg(MULITPLE_FUNCTION_COUNT));
    return false;
  }

  PyMethodDef * methods=new PyMethodDef[2*nAcceptedProperties+nAcceptedMethods+1];

  // Properties
  int methodIndex=0;
  QByteArray doc;
  for(int i=0; i<nProperties; i++) {
    QMetaProperty p=type.qtObject->property(type.propertyOffset+i);
    if(p.isStored()) {
      PyMethodDef& mValue=methods[methodIndex++];
      mValue.ml_name=p.name();
      mValue.ml_flags=METH_NOARGS;
      doc=p.name();
      doc.prepend(" ");
      doc.prepend(p.typeName());
      doc+="(): returns property value.";
      _methodNames.append(doc);
      mValue.ml_doc=_methodNames.last().data();
      mValue.ml_meth=_mapValues[i];
      PyMethodDef& mSetValue=methods[methodIndex++];
      QByteArray name(p.name());
      if(name.at(0)>='a') {
#if(QT_VERSION>=QT_VERSION_CHECK(6, 0, 0))
        name[0]+='A'-'a';
#else
        name[0]=name[0]+static_cast<char>('A'-'a');
#endif
      }
      name.prepend("set");
      _methodNames.append(name);
      mSetValue.ml_name=_methodNames.last().data();
      mSetValue.ml_flags=METH_VARARGS;
      doc=name;
      doc+="(";
      doc+=p.typeName();
      doc+="): sets property value.";
      _methodNames.append(doc);
      mSetValue.ml_doc=_methodNames.last().data();
      mSetValue.ml_meth=_mapSetValues[i];
    }
  }
  // Methods
  for(it=methodSort.begin(); it!=methodSort.end(); it++) {
    int index=it.value();
    QMetaMethod m=type.qtObject->method(type.methodOffset+index);
    PyMethodDef& mMethod=methods[methodIndex++];
    if(methodOverloads[m.name()].count>1) {
      QByteArray name(m.name());
      name+="_";
      name+=QByteArray::number(++methodOverloads[m.name()].index);
      _methodNames.append(name);
      mMethod.ml_name=_methodNames.last().data();
    } else {
      mMethod.ml_name=m.name();
    }
    mMethod.ml_flags=METH_VARARGS;
    QByteArray doc(mMethod.ml_name);
    QList<QByteArray> pnames=m.parameterNames();
    doc+="(";
    for(int i=0; i<m.parameterCount()-1; i++) {
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
      doc+=m.parameterMetaType(i).name();
#else
      doc+=QMetaType::typeName(m.parameterType(i));
#endif
      doc+=" ";
      doc+=pnames.at(i);
      doc+=", ";
    }
    if(m.parameterCount()>0) {
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
      doc+=m.parameterMetaType(m.parameterCount()-1).name();
#else
      doc+=QMetaType::typeName(m.parameterType(m.parameterCount()-1));
#endif
      doc+=" ";
      doc+=pnames.at(m.parameterCount()-1);
    }
    doc+=")";
    if(m.returnType()!=QMetaType::Void) {
      doc.prepend(" ");
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
      doc.prepend(m.returnMetaType().name());
#else
      doc.prepend(QMetaType::typeName(m.returnType()));
#endif
    }
    _methodNames.append(doc);
    mMethod.ml_doc=_methodNames.last().data();
    mMethod.ml_meth=_mapMethods[index];
  }

  ASSERT(methodIndex==2*nAcceptedProperties+nAcceptedMethods);
  methods[methodIndex].ml_name=NULL;

  type.pyObject->tp_methods=methods;

  return PyType_Ready(type.pyObject)>=0;
}

bool PythonClassFactory::isAcceptableMethod(const QMetaMethod& m)
{
  if(!MethodArgument::isSupported(m.returnType())) {
    //printf("  %s %s (%i) not supported as return type\n", m.name().data(),
    //       m.returnMetaType().name(), m.returnMetaType().id());
    return false;
  }
  for(int i=0; i<m.parameterCount(); i++) {
    if(!MethodArgument::isSupported(m.parameterType(i))) {
      //printf("  %s %s (%i) not supported as argument type\n", m.name().data(),
      //       m.parameterMetaType(i).name(), m.returnMetaType().id());
      return false;
    }
  }
  if(m.parameterCount()>MethodArgument::maximumArgumentCount()) {
    //printf("  method %s with more than %i parameters is not supported\n",
    //       m.name().data(), MethodArgument::maximumArgumentCount());
    return false;
  }
  return true;
}

PyObject * PythonClassFactory::value(int objectId, int propertyId)
{
  QVariant val=GraphicalUserInterface::instance()->propertyValue(objectId, propertyId);
  if(val.isValid()) {
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
    switch(val.metaType().id()) {
#else
    switch(val.type()) {
#endif
    case QMetaType::Bool:
      return val.toBool() ? Py_True : Py_False;
    case QMetaType::Int:
      return PyLong_FromLong(val.toInt());
    case QMetaType::Double:
      return PyFloat_FromDouble(val.toDouble());
    case QMetaType::QString: {
        QString valStr=val.toString();
        return PyUnicode_FromKindAndData(PyUnicode_2BYTE_KIND, valStr.data(), valStr.size());
      }
    default:
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
      GraphicalUserInterface::instance()->setErrorMessage(tr("unsupported type %1").arg(val.metaType().id()));
#else
      GraphicalUserInterface::instance()->setErrorMessage(tr("unsupported type %1").arg(val.type()));
#endif
      return NULL;
    }
  } else {
    GraphicalUserInterface::instance()->setErrorMessage(tr("invalid property value"));
    return NULL;
  }
}

PyObject * PythonClassFactory::setValue(int objectId, int propertyId, PyObject * args)
{
  if(GraphicalUserInterface::instance()->setPropertyValue(objectId, propertyId, args)) {
    Py_RETURN_NONE;
  } else {
    return NULL;
  }
}

PyObject * PythonClassFactory::method(int objectId, int methodId, PyObject * args)
{
  MethodArgument * val=GraphicalUserInterface::instance()->method(objectId, methodId, args);
  if(val) {
    return val->toPyObject();
  } else {
    return NULL;
  }
}

int PythonClassFactory::propertyOffset(const QString& className) const
{
  QMap<QString, TypeInfo>::const_iterator it=_types.find(className);
  if(it!=_types.end()) {
    return it.value().propertyOffset;
  } else {
    GraphicalUserInterface::instance()->setErrorMessage(tr("unknown object type '%1' (propertyOffset)")
                                                            .arg(className));
    return -1;
  }
}

int PythonClassFactory::methodOffset(const QString& className) const
{
  QMap<QString, TypeInfo>::const_iterator it=_types.find(className);
  if(it!=_types.end()) {
    return it.value().methodOffset;
  } else {
    GraphicalUserInterface::instance()->setErrorMessage(tr("unknown object type '%1' (method offset)")
                                                            .arg(className));
    return -1;
  }
}

const QMetaObject * PythonClassFactory::metaObject(const QString& className) const
{
  QMap<QString, TypeInfo>::const_iterator it=_types.find(className);
  if(it!=_types.end()) {
    return it.value().qtObject;
  } else {
    GraphicalUserInterface::instance()->setErrorMessage(tr("unknown object type '%1' (metaObject)")
                                                            .arg(className));
    return nullptr;
  }
}

PyObject * PythonClassFactory::create(const QString& className, int objectId) const
{
  QMap<QString, TypeInfo>::const_iterator it=_types.find(className);
  if(it!=_types.end()) {
    PyObject * pyObj=_PyObject_New(it.value().pyObject);
    ((QtObjectData *)pyObj)->objectId=objectId;
    return pyObj;
  } else {
    GraphicalUserInterface::instance()->setErrorMessage(tr("unknown object type '%1' (create)")
                                                            .arg(className));
    return NULL;
  }
}

int PythonClassFactory::objectId(PyObject * obj)
{
  return ((QtObjectData *)obj)->objectId;
}

MULTIPLE_FUNCTION(IMPL_VALUE_FUNCTION)
MULTIPLE_FUNCTION(IMPL_METHOD_FUNCTION)

#endif // HAS_PYTHON
