/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-03-09
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "NormalizationParameters.h"

namespace GeopsyCore {

/*!
  \class NormalizationParameters NormalizationParam.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

  /*!
    Description of constructor still missing
  */
  NormalizationParameters::NormalizationParameters()
    : AbstractParameters()
  {
    TRACE;
    _type=None;
    _fmin=0.5;
    _fmax=50.0;
  }

  /*!
    Description of constructor still missing
  */
  NormalizationParameters::NormalizationParameters(const NormalizationParameters& o)
    : AbstractParameters(o)
  {
    TRACE;
    _type=o._type;
    _fmin=o._fmin;
    _fmax=o._fmax;
  }

  void NormalizationParameters::setType(Type t)
  {
    TRACE;
    _type=t;
    if(_type==SpectrumEnergy) {
      setFrequencyRange(_fmin, _fmax);
    }
  }

  void NormalizationParameters::setType(const QString& t)
  {
    if(t.size()>3) {
      QString lt=t.toLower();
      switch(lt[2].unicode()) {
      case 'x':
        if(lt=="maximumamplitude") {
          _type=MaximumAmplitude;
          return;
        }
        break;
      case 'n':
        if(lt=="none") {
          _type=None;
          return;
        }
        break;
      case 'r':
        if(lt=="surfaceattenuation") {
          _type=SurfaceAttenuation;
          return;
        }
        break;
      case 'e':
        if(lt=="spectrumenergy") {
          _type=SpectrumEnergy;
          setFrequencyRange(_fmin, _fmax);
        }
        break;
      case 'l':
        if(lt=="volumeattenuation") {
          _type=VolumeAttenuation;
          return;
        }
        break;
      default:
        break;
      }
      App::log(tr("Unrecognized normalization type: %1\n").arg(t) );
    }
  }

  QString NormalizationParameters::typeString() const
  {
    switch(_type) {
    case None:
      break;
    case SurfaceAttenuation:
      return "SurfaceAttenuation";
    case VolumeAttenuation:
      return "VolumeAttenuation";
    case MaximumAmplitude:
      return "MaximumAmplitude";
    case SpectrumEnergy:
      return "SpectrumEnergy";
    }
    return "None";
  }

  void NormalizationParameters::setFrequencyRange(double min, double max)
  {
    TRACE;
    if(min<max) {
      _fmin=min;
      _fmax=max;
    } else {
      _fmin=max;
      _fmax=min;
    }
  }

  QString NormalizationParameters::toString(PARAMETERS_TOSTRING_ARGS_IMPL) const
  {
    TRACE;
    QString log;
    log+=prefix+"NORMALIZATION TYPE"+suffix+"="+typeString()+"\n";
    log+=prefix+"NORMALIZATION MINIMUM"+suffix+"="+QString::number(_fmin)+"\n";
    log+=prefix+"NORMALIZATION MAXIMUM"+suffix+"="+QString::number(_fmax)+"\n";
    return log;
  }

  int NormalizationParameters::keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const
  {
    return 3+AbstractParameters::keywordCount();
  }

  void NormalizationParameters::collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS)
  {
    TRACE;
    int baseIndex=AbstractParameters::keywordCount();
    keywords.add(prefix+"NORMALIZATION TYPE"+suffix, this, baseIndex);
    keywords.add(prefix+"NORMALIZATION MINIMUM"+suffix, this, baseIndex+1);
    keywords.add(prefix+"NORMALIZATION MAXIMUM"+suffix, this, baseIndex+2);
  }

  bool NormalizationParameters::setValue(PARAMETERS_SETVALUE_ARGS)
  {
    TRACE;
    switch(index-AbstractParameters::keywordCount()) {
    case 0:
      setType(value);
      return true;
    case 1:
      _fmin=value.toDouble();
      return true;
    case 2:
      _fmax=value.toDouble();
      return true;
    default:
      break;
    }
    return AbstractParameters::setValue(index, value, unit, keywords);
  }

} // namespace GeopsyCore
