/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-05-15
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>

#include "FilterParameters.h"

namespace GeopsyCore {

  /*!
    \class FilterParameters FilterParameters.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  FilterParameters::FilterParameters()
  {
    _fmin=1.0;
    _fmax=1.0;
    _band=LowPass;
    _method=FrequencyWindow;
    _window.setShape(WindowFunctionParameters::Cosine);
    _width=0.1;
  }

  FilterParameters::FilterParameters(const FilterParameters &o)
    : AbstractParameters(o),
      _window(o._window)
  {
    _band=o._band;
    setFrequencyRange(o._fmin, o._fmax);
    _method=o._method;
    _order=o._order;
    _width=o._width;
  }

  void FilterParameters::operator=(const FilterParameters& o)
  {
    AbstractParameters::operator=(o);
    _window=o._window;
    _band=o._band;
    setFrequencyRange(o._fmin, o._fmax);
    _method=o._method;
    _order=o._order;
    _width=o._width;
  }

  void FilterParameters::setBand(Band b)
  {
    _band=b;
    setFrequencyRange(_fmin, _fmax);
    _window.setReversed(_band==BandReject);
  }

  void FilterParameters::setFrequencyRange(double min, double max)
  {
    TRACE;
    switch(_band) {
    case LowPass:
    case HighPass:
      _fmin=min;
      _fmax=min;
      break;
    case BandPass:
    case BandReject:
      if(min<max) {
        _fmin=min;
        _fmax=max;
      } else {
        _fmin=max;
        _fmax=min;
      }
      break;
    }
  }

  ENUM_AS_STRING_BEGIN(FilterParameters, Band)
  ENUM_AS_STRING_DATA_4(BandPass, BandReject, HighPass, LowPass);
  ENUM_AS_STRING_SYNONYM("band pass", BandPass);
  ENUM_AS_STRING_SYNONYM("band reject", BandReject);
  ENUM_AS_STRING_END

  ENUM_AS_STRING_BEGIN(FilterParameters, Method)
  ENUM_AS_STRING_DATA_2(Butterworth, FrequencyWindow);
  ENUM_AS_STRING_SYNONYM("Convolution", FrequencyWindow);
  ENUM_AS_STRING_SYNONYM("CosineTaper", FrequencyWindow);
  ENUM_AS_STRING_END

  void FilterParameters::setVersion(PARAMETERS_SETVERSION_ARGS)
  {
    TRACE;
    _window.setVersion(version);
    AbstractParameters::setVersion(version);
  }

  int FilterParameters::keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const
  {
    return 7+AbstractParameters::keywordCount();
  }

  int FilterParameters::totalKeywordCount(PARAMETERS_TOTALKEYWORDCOUNT_ARGS) const
  {
    return AbstractParameters::totalKeywordCount()
        +_window.totalKeywordCount();
  }

  void FilterParameters::collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS)
  {
    TRACE;
    int baseIndex=AbstractParameters::keywordCount();
    keywords.add(prefix+"FILTER_TYPE"+suffix, this, baseIndex);
    keywords.add(prefix+"FILTER_METHOD"+suffix, this, baseIndex+1);
    keywords.add(prefix+"FILTER_MIN_FREQUENCY"+suffix, this, baseIndex+2);
    keywords.add(prefix+"FILTER_MAX_FREQUENCY"+suffix, this, baseIndex+3);
    keywords.add(prefix+"FILTER_CAUSAL"+suffix, this, baseIndex+4);
    keywords.add(prefix+"FILTER_ORDER"+suffix, this, baseIndex+5);
    keywords.add(prefix+"FILTER_WIDTH"+suffix, this, baseIndex+6);
    _window.collectKeywords(keywords, prefix+"FILTER_", suffix);
  }

  bool FilterParameters::setValue(PARAMETERS_SETVALUE_ARGS)
  {
    TRACE;
    bool ok=true;
    switch(index-AbstractParameters::keywordCount()) {
    case 0:
      setBand(convertBand(value, ok));
      return ok;
    case 1:
      setMethod(convertMethod(value, ok));
      return ok;
    case 2:
      _fmin=value.toDouble(&ok);
      return ok;
    case 3:
      _fmax=value.toDouble(&ok);
      return ok;
    case 4:
      setCausal(value=="y");
      return true;
    case 5:
      setOrder(value.toInt(&ok));
      return ok;
    default:
      break;
    }
    return AbstractParameters::setValue(index, value, unit, keywords);
  }

  QString FilterParameters::toString(PARAMETERS_TOSTRING_ARGS_IMPL) const
  {
    TRACE;
    QString log;
    log+=prefix+"FILTER_TYPE"+suffix+"="+convertBand(_band)+"\n";
    log+=prefix+"FILTER_METHOD"+suffix+"="+convertMethod(_method)+"\n";
    log+=prefix+"FILTER_MIN_FREQUENCY"+suffix+" (Hz)="+QString::number(_fmin)+"\n";
    log+=prefix+"FILTER_MAX_FREQUENCY"+suffix+" (Hz)="+QString::number(_fmax)+"\n";
    switch(_method) {
    case Butterworth:
      log+=prefix+"FILTER_CAUSAL"+suffix+" (y/n)="+(causal() ? "y" : "n")+"\n";
      log+=prefix+"FILTER_ORDER"+suffix+"="+QString::number(_order)+"\n";
      break;
    case FrequencyWindow:
      log+=_window.toString(prefix+"FILTER_", suffix);
      log+=prefix+"FILTER_WIDTH"+suffix+"="+QString::number(_width);
      break;
    }
    return log;
  }

} // namespace GeopsyCore
