/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-09-18
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <signal.h>

#include "CoreToolInterface.h"
#include "SignalDatabase.h"
#include "SignalGroup.h"
#include "AbstractTool.h"

namespace GeopsyCore {

   AbstractTool * CoreToolInterface::_tool=nullptr;

  /*!
    \class CoreToolInterface CoreToolInterface.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Generally a global object initialized before CoreApplication.
    If default outputFileName is something else than 'a.max',
    use setOuputFileName() before initializing CoreApplication
  */
  CoreToolInterface::CoreToolInterface()
  {
    TRACE;
    _param=nullptr;
    _overwrite=true;
  }

  /*!
    Description of destructor still missing
  */
  CoreToolInterface::~CoreToolInterface()
  {
    TRACE;
    delete _param;
  }

  void CoreToolInterface::setParameters(AbstractParameters * param)
  {
    TRACE;
    delete _param;
    _param=param;
  }

  bool CoreToolInterface::setOptions(int& argc, char ** argv)
  {
    App::setPrefix(CoreApplication::applicationName()+": ");
    CoreApplication::instance()->debugUserInterrupts(false);
#ifdef Q_OS_WIN
    signal(30, showProgress);  // From the list available with 'kill -l'
#else
    signal(SIGUSR1, showProgress);
    signal(SIGINT, stopAndSave);
#endif

    QString databaseFileName;
    QStringList groupPaths;
    QStringList groupPatterns;
    QString utmCoordinateFile;
    QString saveDatabase;
    QList<int> groupIds;
    bool showGroups=false, showCoord=false, groupLeaves=false;

    int j=1;
    for (int i=1; i<argc; i++) {
      QByteArray arg = argv[i];
      if (arg[0]=='-') {
        if(arg=="-db") {
          CoreApplication::checkOptionArg(i, argc, argv);
          if(!databaseFileName.isEmpty()) {
            App::log(tr("only one option '-db' is allowed\n") );
            return false;
          }
          databaseFileName=argv[i];
        } else if(arg=="-no-file-key") {
          GeopsyCoreEngine::instance()->setFileContentKey(false);
        } else if(arg=="-group-leaves") {
          groupLeaves=true;
        } else if(arg=="-groups") {
          showGroups=true;
        } else if(arg=="-group" || arg=="-group-path") {
          CoreApplication::checkOptionArg(i, argc, argv);
          groupPaths.append(argv[i]);
        } else if(arg=="-group-pattern") {
          CoreApplication::checkOptionArg(i, argc, argv);
          groupPatterns.append(argv[i]);
        } else if(arg=="-group-id") {
          CoreApplication::checkOptionArg(i, argc, argv);
          groupIds.append(CoreApplication::toInt(i, i-1, argv));
        } else if(arg=="-o") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _outputBaseName=argv[i];
        } else if(arg=="-station-list") {
          CoreApplication::checkOptionArg(i, argc, argv);
          QString l(argv[i]);
#if(QT_VERSION >= QT_VERSION_CHECK(5, 14, 0))
          _stationList=l.split(QRegularExpression("[, ]"), Qt::SkipEmptyParts);
#else
          _stationList=l.split(QRegularExpression("[, ]"), QString::SkipEmptyParts);
#endif
        } else if(arg=="-coord") {
          showCoord=true;
        } else if(arg=="-utm") {
          CoreApplication::checkOptionArg(i, argc, argv);
          utmCoordinateFile.append(argv[i]);
        } else if(arg=="-save-db") {
          CoreApplication::checkOptionArg(i, argc, argv);
          saveDatabase=argv[i];
        } else if(arg=="-param") {
          CoreApplication::checkOptionArg(i, argc, argv);
          QStringList beginPatterns, endPatterns;
          beginPatterns << "# BEGIN PARAMETERS";
          endPatterns << "# END PARAMETERS";
          if(!_param->load(argv[i], beginPatterns, endPatterns)) {
            App::log(tr("error loading parameters from '%1', see '-param-example'\n").arg(argv[i]) );
            return false;
          }
        } else if(arg=="-set") {
          CoreApplication::checkOptionArg(i, argc, argv);
          if(!_param->parse(argv[i])) {
            App::log(tr("Error setting parameter '%1'.\n").arg(argv[i]) );
            return false;
          }
        } else if(arg=="-fix-signal-paths") {
          CoreApplication::instance()->setBatch(false);
          if(databaseFileName.isEmpty()) {
            App::log(tr("Option '-db' must be defined before '-fix-signal-path'.\n") );
            return false;
          }
          ::exit(fixDatabasePaths(databaseFileName) ? 0 : 2);
        } else if(arg=="-no-overwrite") {
          _overwrite=false;
        } else {
          argv[j++]=argv[i];
        }
      } else {
        argv[j++]=argv[i];
      }
    }
    if(j<argc) {
      argv[j]=nullptr;
      argc=j;
    }

    QStringList importedSignals=CoreApplication::getFileList(argc, argv);
    SignalDatabase * db=GeopsyCoreEngine::instance()->defaultDatabase();
    if(!databaseFileName.isEmpty() && !importedSignals.isEmpty()) {
      App::log(tr("a database and signals to import cannot be specified at the same time.\n") );
      return false;
    }
    db->load(databaseFileName, QString(), importedSignals, QString(), false);
    if(db->isEmpty()) {
      App::log(tr("no database specified or no signals to import, see -help\n") );
      return false;
    }
    if(showGroups) {
      db->masterGroup()->printList(QString());
      ::exit(0);
    }
    if(!groupIds.isEmpty()) {
      for(QList<int>::iterator it=groupIds.begin(); it!=groupIds.end(); it++) {
        AbstractSignalGroup * g=db->findGroup(*it);
        if(!g) {
          App::log(tr("group id %1 does not exist\n").arg(*it) );
          return false;
        }
        if(groupLeaves) {
          _groups.append(g->list());
        } else {
          _groups.append(g);
        }
      }
    }
    if(!groupPaths.isEmpty()) {
      for(QStringList::iterator it=groupPaths.begin(); it!=groupPaths.end(); it++) {
        AbstractSignalGroup * g=db->findGroup(*it);
        if(!g) {
          App::log(tr("group '%1' does not exist\n").arg(*it) );
          return false;
        }
        if(groupLeaves) {
          _groups.append(g->list());
        } else {
          _groups.append(g);
        }
      }
    }
    if(!groupPatterns.isEmpty()) {
      for(QStringList::iterator it=groupPatterns.begin(); it!=groupPatterns.end(); it++) {
        QList<const AbstractSignalGroup *> list=const_cast<const SignalDatabase *>(db)->findGroups(QRegularExpression(*it));
        if(list.isEmpty()) {
          App::log(tr("group pattern '%1' has no match\n").arg(*it) );
          return false;
        }
        if(groupLeaves) {
          for(int i=0; i<list.count(); i++) {
            _groups.append(list.at(i));
          }
        } else {
          _groups.append(list);
        }
      }
    }
    if(_groups.isEmpty()) {
      if(!groupIds.isEmpty() ||
         !groupPaths.isEmpty() ||
         !groupPatterns.isEmpty()) {
        App::log(tr("Empty group selection\n") );
        return false;
      }
      _groups=db->masterGroup()->list();
    }
    if(!utmCoordinateFile.isEmpty()) {
      App::log(tr("Setting coordinates from '%1'...\n").arg(utmCoordinateFile));
      if(!setUtmCoordinates(utmCoordinateFile)) {
        return false;
      }
    }
    if(!saveDatabase.isEmpty()) {
      SignalGroup * g=new SignalGroup;
      g->setSignals(db->subPool());
      g->setName("all");
      App::log(tr("Creating group 'all' with all loaded signals\n"));
      g->setParent(db->masterGroup());
      if(db->saveAs(saveDatabase)) {
        App::log(tr("Database saved to '%1'\n").arg(saveDatabase));
      }
      ::exit(0);
    }
    App::log(tr("Selected groups:\n"));
    for(QList<const AbstractSignalGroup *>::iterator it=_groups.begin(); it!=_groups.end(); it++) {
      App::log(tr("  '%1': %2 signals\n").arg((*it)->pathName()).arg((*it)->signalCount()) );
    }
    if(showCoord) {
      QTextStream sOut(stdout);
      int n=subPoolCount();
      for(int i=0; i<n; i++) {
        SubSignalPool p=subPool(i);
        QList<NamedPoint> l=p.stations();
        sOut << "# " << groupName(i) << Qt::endl;
        for(QList<NamedPoint>::iterator its=l.begin(); its!=l.end(); its++) {
          sOut << its->toString('g', 20) << Qt::endl;
        }
      }
      ::exit(0);
    }
    return true;
  }

  bool CoreToolInterface::setUtmCoordinates(const QString& utmCoordinateFile)
  {
    CoordinateFile cf;
    cf.setUtmXYNameParser();
    if(!cf.setFile(utmCoordinateFile)) {
      return false;
    }
    while(!cf.wait()) {}
    QList<NamedPoint> coordinates=cf.points();
    // Populate map for quick lookup.
    QMap<QString, int> coordinateMap;
    for(int i=coordinates.count()-1; i>=0; i--) {
      coordinateMap.insert(coordinates.at(i).name(), i);
    }
    SignalDatabase * db=GeopsyCoreEngine::instance()->defaultDatabase();
    QMap<QString, int>::iterator it;
    for(int i=db->count()-1; i>=0; i--) {
      Signal * sig=db->at(i);
      it=coordinateMap.find(sig->name());
      if(it!=coordinateMap.end()) {
        const NamedPoint& p=coordinates.at(it.value());
        App::log(tr("Set coordinates (%1) to signal %2 (ID=%3)\n")
                 .arg(p.toString('f', 3))
                 .arg(sig->nameComponent())
                 .arg(sig->id()));
        sig->setReceiver(p);
        sig->setUtmZone(cf.utmZone());
      }
    }
    return true;
  }

  void CoreToolInterface::showArguments()
  {
    App::log(CoreApplication::applicationFilePath()+" "+
             CoreApplication::argumentList().join(' ')+"\n");
  }

  int CoreToolInterface::subPoolCount() const
  {
    TRACE;
    if(_groups.isEmpty()) {
      return 1;
    } else {
      return _groups.count();
    }
  }

  /*!
    Return subpool corresponding to group \a index. Excluded stations are removed from the returned subpool.
  */
  SubSignalPool CoreToolInterface::subPool(int index) const
  {
    TRACE;
    SubSignalPool p;
    if(_groups.isEmpty()) {
      p.addAll(GeopsyCoreEngine::instance()->defaultDatabase());
    } else {
      p.addGroup(_groups.at(index));
    }
    if(!_stationList.isEmpty()) {
      p.remove(_stationList);
    }
    return p;
  }

  QString CoreToolInterface::groupName(int index) const
  {
    if(_groups.isEmpty()) {
      return "importedsignals";
    } else {
      return _groups.at(index)->name();
    }
  }

  bool CoreToolInterface::fixDatabasePaths(QString databaseFileName)
  {
    TRACE;
    SignalDatabase * db=GeopsyCoreEngine::instance()->defaultDatabase();
    if(!db->open(databaseFileName)) {
      return false;
    }
    return db->save();
  }

  bool CoreToolInterface::overwrite(const QString& fileName) const
  {
    if(_overwrite || !QFile::exists(fileName)) {
      return true;
    } else {
      App::log(tr("File '%1' already exists.\n").arg(fileName));
      return false;
    }
  }

  void CoreToolInterface::showProgress(int)
  {
    if(_tool) {
      _tool->showProgress();
    }
  }

  void CoreToolInterface::stopAndSave(int)
  {
    if(_tool) {
      _tool->stop();  // If one of the worker is blocked, it will not necessarily respond to this stop
                     // Hence we save what can be saved.
      signal(SIGINT, nullptr);
      App::log(tr("Waiting for the termination during 60 sec. maximum\n"
                  "The collected results will be saved even if some tasks are blocked.\n"
                  "Send another interrupt signal to skip saving results.\n"));
      _tool->waitFinished(60000);
      if(!_tool->isFinished()) {
        // The process did not stop properly after 60 seconds.
        QString fileName=File::uniqueName("backup.max", QDir());
        _tool->saveResults(fileName);
        App::log(tr("Some tasks are blocked. Aborting\n"));
        std::exit(2);
      }
      App::log(tr("Clean termination of all tasks achieved\n"
                  "Waiting 10 sec. before saving the results.\n"
                  "Send another interrupt signal to skip saving results.\n"));
      CoreApplication::sleep(10000);
    }
  }

  void CoreToolInterface::setHelp(ApplicationHelp * h)
  {
    h->addOption("-db", "Database file to process if no signal files are imported.");
    h->addOption("-save-db <FILE.gpy>", "Creates database FILE.gpy from the loaded signals and exits.");
    h->addOption("-no-file-key", "Skip file content key when loading database (only for old database before 2017-09).");
    h->addOption("-interactive", "Allows interactions through stdin/out.");
    h->addOption("-no-overwrite", "If output file already exists, does not process.");
    h->addOption("-groups", "Lists the available groups and exits.");
    h->addOption("-group-leaves", "When a group is selected by one of the following options, \"leaves\" means: process "
                                  "individualy all elementary children. By default, the selected folder groups are process "
                                  "in one block. This option has no effect if the selected groups are already elementary groups "
                                  " with no children.");
    h->addOption("-group-id <ID>", "Group ID to process. Several entries are allowed. "
                                   "If no group is selected, all groups are processed. If group with ID contains several children, "
                                   "the children are processed.");
    h->addOption("-group <PATH>", "Alias to -group-path.");
    h->addOption("-group-path <PATH>", "Group path to process. Several entries are allowed. "
                                       "If no group is selected, all groups are processed. If group with PATH contains several children, "
                                       "the children are processed.");
    h->addOption("-group-pattern <REGEXP>", "Select for processing all groups whose name matches REGEXP. Only one entry is allowed. "
                                            "If no group is selected, all groups are processed.");
    h->addOption("-station-list <STATIONS>", "STATIONS is a coma separated list of station names to be processed. "
                                             "'-group-*' options are still required but only a subset of the group "
                                             "can be selected with option '-station-list'.");
    h->addOption("-utm <FILE>", "If provided, the coordinates of stations are assigned from the UTM coordinates in file. "
                                "File columns must be: UTM_ZONE X Y STATION_NAME. Use UTM_ZONE as '---' for cartesian non UTM coordinates.");
    h->addOption("-coord", "Returns the list of selected stations and their coordinates.");
    h->addOption("-o <FILE>", "Ouput file name (default='a')");
    h->addOption("-param <PARAM>", "Load parameters in PARAM");
    h->addOption("-set <PARAM>=<VALUE>", "Set VALUE to parameter PARAM.");
    h->addOption("-fix-signal-paths", "Only '-db' option is required. Absolute paths stored in a '.gpy' can be changed "
                                      "without opening a graphical user interface. Useful when moving a database to a "
                                      "remote server without X display.");
  }

} // namespace GeopsyCore

