/***************************************************************************
**
**  This file is part of DinverDCCore.
**
**  DinverDCCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverDCCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2010-04-02
**  Copyright: 2010-2019
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef TARGETLIST2D_H
#define TARGETLIST2D_H

#include "TargetList.h"

namespace DinverDCCore {

class Param3DGroundModel;

class DINVERDCCORE_EXPORT TargetList2D : public XMLClass
{
public:
  TargetList2D();
  TargetList2D(const TargetList2D& o);
  ~TargetList2D();

  virtual const QString& xml_tagName() const {return xmlTargetList2DTag;}
  static const QString xmlTargetList2DTag;

  void addTarget(TargetList * t);

  int targetCount() const {return _targets.count();}
  TargetList * target(int index) const {return _targets.at(index);}

  bool setParamProfiles(Param3DGroundModel * gm, RealSpace& ps);
  inline bool isPoissonRatioOk(const GroundParameter * from);
  void setVp();
  double misfit(int nDimensions, bool& ok);
  void writeReport(ReportWriter * report) const;
  void humanInfo() const;
protected:
  virtual void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
private:
  friend class DCReportBlock;

  void interpole(const Point& at, const Profile& p1, const Profile& p2, const Profile& p3, Profile& p);

  ParamProfile * _vp[3];
  ParamProfile * _nu[3];
  ParamProfile * _vs[3];
  ParamProfile * _rho[3];
  PoissonCondition * _poissonCondition[3];
  QHash<const ParamGroundModel *, int> _models;

  VectorList<TargetList *> _targets;

  Plane _planeEngine;
  Profile _reportVp, _reportVs, _reportRho;
};

inline bool TargetList2D::isPoissonRatioOk(const GroundParameter * from)
{
  TRACE;
  QHash<const ParamGroundModel *, int>::iterator it=_models.find(from->groundModel());
  ASSERT(it!=_models.end());
  PoissonCondition * pc=_poissonCondition[it.value()];
  if(pc) {
    return pc->isOk(from);
  } else {
    return true;
  }
}

} // namespace DinverDCCore

#endif // TARGETLIST2D_H
