/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-06
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>

#include "AbstractForward.h"
#include "ReportWriter.h"

namespace DinverCore {

  /*!
    \class AbstractForward AbstractForward.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Copy current values from \a o. Used during the initialization of several
    parallel forward threads (intialized with the same values).
  */
  void AbstractForward::copyValues(const AbstractForward& o)
  {
    TRACE;
    int ndVar=_parameterSpace.variableParameterCount();
    for(int i=0; i<ndVar; i++) {
      Parameter * p=_parameterSpace.variableParameter(i);
      p->setRealValue(o.parameterSpace().variableParameter(i)->realValue());
    }
    valueChanged();
  }

  /*!
    Provided for convenience, it calls misfit(double *, bool) to compute misfit from a vector
    of parameter values. Reimplement this function if you do not need values under this
    format (direct access to parameter space).
  */
  double AbstractForward::misfit(bool& ok)
  {
    int ndAll=_parameterSpace.allParameterCount();
    double model[ndAll];
    for(int i=0;i<ndAll;i++) {
      model[i]=_parameterSpace.parameter(i)->realValue();
    }
    return misfit(model, ok);
  }

  /*!
    Generates a very first model that satisfy all conditions.
  */
  bool AbstractForward::firstModel(Random& randomNumbers, AtomicBoolean& terminated)
  {
    TRACE;
    // Generate very first model
    // Set random values for parameters without checking conditions
    int ndVar=_parameterSpace.variableParameterCount();
    for(int i=0; i<ndVar; i++) {
      Parameter * p=_parameterSpace.variableParameter(i);
      p->setGridValue(randomNumbers.uniform(0, p->gridCount()));
    }
    App::log(tr("   The current model is probably not yet inside the parameter space.\n"
                "   Some warnings here can be ignored.\n"));
    // Some parametrizations may need some updates before proceeding
    valueChanged();

    // Iterate untill finding a valid model (fussy parameters conditions are not
    // handled because the current model may not be totally valid)
    int nWalks=1;
    int nWalksDiagnostic=64;
    int minp, maxp;
    SAFE_UNINITIALIZED(minp, 0);
    SAFE_UNINITIALIZED(maxp, 0);
    while(!_parameterSpace.isOk()) {
      if(nWalks==nWalksDiagnostic) {
        App::log(tr("   Random model generated from scratch: not possible to find a good model after %1 walks.\n").arg(nWalks) );
        _parameterSpace.conditionDiagnostic();
        nWalksDiagnostic*=2;
      }
      nWalks++;
      for(int i=0;i<ndVar; i++) {
        Parameter * p=_parameterSpace.variableParameter(i);
        p->getGridLimits(minp, maxp);
        //double dmin, dmax;
        //p->getRectangularLimits(dmin, dmax);
        //p->getLimits(dmin, dmax);
        //printf("Parameter %s from %lf to %lf",p->name().toLatin1().data(), dmin, dmax);
        if(maxp<=minp) { // Forced to a fixed parameter, break it, back to initial range
          //printf("Back to original range for parameter %s\n",p->name().toLatin1().data());
          minp=0;
          maxp=p->gridCount();
        }
        p->setGridValue(randomNumbers.uniform(minp, maxp));
        //printf("  ---> %lf\n", p->realValue());
        // Some parametrizations may need some updates before proceeding
        valueChanged(p);
      }
      //_parameterSpace.printModel();
      if(terminated.value()) {
        return false;
      }
    }
    //unlock();
    App::log(tr("   Random model generated from scratch: perfomed %1 random walk(s)\n").arg(nWalks) );
    //printModel();
    return true;
  }

  /*!
    Generates a new valid random model by introducing random perturbations to the
    current values.
    Returns true if generation is fine and a misfit can be computed.
  */
  bool AbstractForward::randomModel(Random& randomNumbers, int walkCount)
  {
    TRACE;

    int ndVar=_parameterSpace.variableParameterCount();
    ASSERT(_parameterSpace.isOkDebug());
    int minp, maxp;
    SAFE_UNINITIALIZED(minp, 0);
    SAFE_UNINITIALIZED(maxp, 0);

    // Some parametrizations may need some updates before proceeding
    valueChanged();

//#define RANDOM_MODEL_DEBUG

    // Makes a minimum number of random walks to provide a statistically
    // independent model (from the current one)
    for(int iw=0; iw<walkCount; iw++) {
      for(int i=0; i<ndVar; i++) {
        Parameter * p=_parameterSpace.variableParameter(i);
#ifdef RANDOM_MODEL_DEBUG
        App::log(tr("randomWalk: parameter[%1]:%2\n").arg(i).arg(p->name()));
        App::log(_parameterSpace.toString()+"\n");
#endif
        p->getGridLimits(minp, maxp);
#ifdef RANDOM_MODEL_DEBUG
        App::log(tr(" --> from %1 to %2\n").arg(minp).arg(maxp));
#endif
        if(maxp<minp) {
          App::log(tr("Generating random model with random walk\n") );
          App::log(tr("Parameter %1: bad parameter range [%2, %3]\n").arg(p->name())
              .arg(p->realValue(minp)).arg(p->realValue(maxp)));
          App::log(_parameterSpace.toString()+"\n");
          return false;
        } else if(minp<maxp) {
          if(p->isFussy()) {
            int nTries;
            // Allow for a maximum of 100 tries before giving up
            // This is not really a good solution but in some cases it avoids the process to be blocked
            for(nTries=0; nTries<100; nTries++) {
              p->setGridValue(randomNumbers.uniform(minp, maxp) );
              // Some parametrizations may need some updates before proceeding
              valueChanged(p);
              if(isFussyOk(p)) {
                /*if(nTries>0) {
                  App::log(tr("Fussy parameter accepted after %1 tries\n").arg(nTries) );
                }*/
                break;
              }
            }
            if(nTries==100) {
              App::log(tr("Fussy parameter rejected after %1 tries, trying another model.\n").arg(nTries) );
              return false;
            }
          } else {
            p->setGridValue(randomNumbers.uniform(minp, maxp));
            // Some parametrizations may need some updates before proceeding
            valueChanged(p);
          }
        } else {
          p->setGridValue(minp);
          // Some parametrizations may need some updates before proceeding
          valueChanged(p);
        }
#ifdef RANDOM_MODEL_DEBUG
        App::log(tr(" \t---> grid %1 real %2\n").arg(p->gridValue()).arg(p->realValue()));
        ASSERT(_forward->isOk());
        if(fabs(_parameterSpace.variableParameter(1)->realValue()-234.5)<0.1 &&
           fabs(_parameterSpace.variableParameter(2)->realValue()-2.78677)<0.00001) {
          printf("\n");
        }
        ASSERT(_parameterSpace.isOkDebug());
#endif
      }
    }
    return true;
  }

} // namespace DinverCore
