/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-09-08
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "WavefieldValues.h"

namespace ArrayCore {

  void WavefieldValues::Observations::init()
  {
    invObsPowerX=1.0/powerX;
    invObsKConcavityX=1.0/kConcavityX;
    invObsPowerH=1.0/powerH;
    invObsKConcavityH=1.0/kConcavityH;
    invObsPowerZ=1.0/powerZ;
    invObsKConcavityZ=1.0/kConcavityZ;
  }

  WavefieldValues::SteeringMode::SteeringMode()
  {
    _powerEngine=nullptr;
    _dpowerEngine=nullptr;
    _xiSlope=0.0;
    _xiConcavity=0.0;
    _kConcavity=0.0;
    _diffPower=0.0;
    _diffKConcavity=0.0;
    _derPower=0.0;
    _derKSlope=0.0;
    _derKConcavity=0.0;
    _derXiSlope=0.0;
    _derXiConcavity=0.0;
  }

  WavefieldValues::SteeringMode::~SteeringMode()
  {
    delete _powerEngine;
    delete _dpowerEngine;
  }

  void WavefieldValues::SteeringMode::setSteering(const FKSteering * s)
  {
    _powerEngine=new FKPower(s->array()->count(), 3);
    _dpowerEngine=new FKPower(s->array()->count(), 3);
  }

  void WavefieldValues::SteeringMode::setCrossSpectrum(const ComplexMatrix * f)
  {
    _powerEngine->setCrossSpectrum(f);
  }

  void WavefieldValues::SteeringMode::setDerivativeCrossSpectrum(const ComplexMatrix * dfInv)
  {
    _dpowerEngine->setCrossSpectrum(dfInv);
  }

  void WavefieldValues::SteeringMode::setPowerX(const Complex& ell)
  {
    const Vector<Complex>& e=FKSteering::threeComponentRayleighCross(ell, _powerEngine);
    _powerEngine->setHighResolutionValue(e);
  }

  void WavefieldValues::SteeringMode::setPowerH(const Complex& ell)
  {
    const Vector<Complex>& e=FKSteering::threeComponentRayleighRadial(ell, _powerEngine);
    _powerEngine->setHighResolutionValue(e);
  }

  void WavefieldValues::SteeringMode::setPowerZ(const Complex& ell)
  {
    const Vector<Complex>& e=FKSteering::threeComponentRayleighVertical(ell, _powerEngine);
    _powerEngine->setHighResolutionValue(e);
  }

  void WavefieldValues::SteeringMode::setXiSlopeX(const Complex& ell)
  {
    double p=_powerEngine->value();
    const Vector<Complex>& de=FKSteering::threeComponentRayleighCrossGradient(ell, _powerEngine);
    _xiSlope=-2.0*p*p*_powerEngine->ellipticityGradient(de);
  }

  void WavefieldValues::SteeringMode::setXiSlopeH(const Complex& ell)
  {
    double p=_powerEngine->value();
    const Vector<Complex>& de=FKSteering::threeComponentRayleighRadialGradient(ell, _powerEngine);
    _xiSlope=-2.0*p*p*_powerEngine->ellipticityGradient(de);
  }

  void WavefieldValues::SteeringMode::setXiSlopeZ(const Complex& ell)
  {
    double p=_powerEngine->value();
    const Vector<Complex>& de=FKSteering::threeComponentRayleighVerticalGradient(ell, _powerEngine);
    _xiSlope=-2.0*p*p*_powerEngine->ellipticityGradient(de);
  }

  void WavefieldValues::SteeringMode::setXiConcavityX()
  {
    double p=_powerEngine->value();
    _xiConcavity=_powerEngine->xiSecondDerivative();
    _xiConcavity=2.0*(_xiSlope*_xiSlope/p+p-p*p*_xiConcavity);
  }

  void WavefieldValues::SteeringMode::setXiConcavityH(const Complex& ell)
  {
    double p=_powerEngine->value();
    double tan=ell.im()/ell.re();
    _xiConcavity=_powerEngine->xiSecondDerivative();
    _xiConcavity=2.0*(_xiSlope*_xiSlope/p+tan*_xiSlope-p*p*_xiConcavity);
  }

  void WavefieldValues::SteeringMode::setXiConcavityZ(const Complex& ell)
  {
    double p=_powerEngine->value();
    double cot=ell.re()/ell.im();
    _xiConcavity=_powerEngine->xiSecondDerivative();
    _xiConcavity=2.0*(_xiSlope*_xiSlope/p+cot*_xiSlope-p*p*_xiConcavity);
  }

  void WavefieldValues::SteeringMode::setKConcavity()
  {
    _powerEngine->highResolutionKRadialFirstDerivative(_powerEngine->ellipticitySteeringVector());
    _kConcavity=_powerEngine->highResolutionKRadialSecondDerivative();
  }

  double WavefieldValues::SteeringMode::misfit(const double& obsPower, const double& invObsPower,
                                               const double& obsKConcavity, const double& invObsKConcavity)
  {
    _diffPower=(_powerEngine->value()-obsPower)*invObsPower;
    _diffKConcavity=(_kConcavity-obsKConcavity)*invObsKConcavity;
    return _xiSlope*_xiSlope
          +_diffPower*_diffPower
          +_diffKConcavity*_diffKConcavity;
  }

  QString WavefieldValues::SteeringMode::detailedMisfit(char type) const
  {
    static const int width=12;
    QString fmt("  %1 %2 %3 %4 %5 :: %6 %7 %8 %9\n");
    return fmt.arg(type)
          .arg(_powerEngine->value(), width, 'g', 6, QChar(' '))
          .arg(_xiSlope, width, 'g', 6, QChar(' '))
          .arg(_kConcavity, width, 'g', 6, QChar(' '))
          .arg(_xiConcavity, width, 'g', 6, QChar(' '))
          .arg(_diffPower*_diffPower, width, 'g', 6, QChar(' '))
          .arg(_xiSlope*_xiSlope, width, 'g', 6, QChar(' '))
          .arg(_diffKConcavity*_diffKConcavity, width, 'g', 6, QChar(' '))
          .arg(_diffXiConcavity*_diffXiConcavity, width, 'g', 6, QChar(' '));
  }

  void WavefieldValues::SteeringMode::setPowerDerivative()
  {
    double p=_powerEngine->value();
    _derPower=-p*p
        *_dpowerEngine->steering(_powerEngine->ellipticitySteeringVector()).re();
  }

  void WavefieldValues::SteeringMode::setKSlopeDerivative()
  {
    double p=_powerEngine->value();
    _dpowerEngine->sensorProjections().copyValues(_powerEngine->sensorProjections());
    _dpowerEngine->setValue(p);
    _derKSlope=_dpowerEngine->highResolutionKRadialFirstDerivative(_powerEngine->ellipticitySteeringVector());
    _derKSlope+=2.0*_derPower*_powerEngine->highResolutionKRadialFirstDerivative()/p;
  }

  void WavefieldValues::SteeringMode::setKConcavityDerivative()
  {
    double slope=_powerEngine->highResolutionKRadialFirstDerivative();
    double slope2=slope*slope;
    double invPower=1.0/_powerEngine->value();
    double dslope=_dpowerEngine->highResolutionKRadialFirstDerivative();
    _derKConcavity=-2.0*_derPower*slope2*invPower*invPower
        +4.0*invPower*slope*_derKSlope
        +2.0*invPower*_derPower*(_kConcavity-2.0*invPower*slope2)
        +_dpowerEngine->highResolutionKRadialSecondDerivative()-2.0*invPower*dslope*dslope;
  }

  void WavefieldValues::SteeringMode::setXiSlopeDerivative()
  {
    double p=_powerEngine->value();
    _derXiSlope=-2.0*p*p*_dpowerEngine->ellipticityGradient(_powerEngine->ellipticityDerivativeSteeringVector());
    _derXiSlope+=2.0*_derPower*_xiSlope/p;
  }

  double WavefieldValues::SteeringMode::misfitDerivative(const double& invObsPower,
                                                         const double& invObsKConcavity)
  {
    return _xiSlope*_derXiSlope
          +_diffPower*_derPower*invObsPower
          +_diffKConcavity*_derKConcavity*invObsKConcavity;
  }

  /*!
    \class WavefieldValues WavefieldValues.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  WavefieldValues::WavefieldValues()
    : _k(3)
  {
    _steering=nullptr;
  }

  /*!
    Description of destructor still missing
  */
  WavefieldValues::~WavefieldValues()
  {
  }

  void WavefieldValues::setSteering(const FKSteering * s)
  {
    _steering=s;
    _x.setSteering(s);
    _h.setSteering(s);
    _z.setSteering(s);
  }

  void WavefieldValues::setWaveNumber(const Complex& k)
  {
    // minus in front of imaginary part to transform a product
    // of complex numbers into a scalar product (the real part)
    // (a+jb)*(c-jd)=(ac+bd)+j(bc-ad)
    _k[0]=k.re();
    _k[1]=-k.im();
  }

  void WavefieldValues::setCrossSpectrum(const ComplexMatrix * f)
  {
    _x.setCrossSpectrum(f);
    _h.setCrossSpectrum(f);
    _z.setCrossSpectrum(f);
  }

  void WavefieldValues::setSteering()
  {
    _steering->threeComponentRayleigh(_k, _x.powerEngine());
    _h.powerEngine()->steeringVector().copyValues(_x.powerEngine()->steeringVector());
    _z.powerEngine()->steeringVector().copyValues(_x.powerEngine()->steeringVector());
    _steering->threeComponentSensorRadialProjections(_k, _x.powerEngine());
    _h.powerEngine()->sensorProjections().copyValues(_x.powerEngine()->sensorProjections());
    _z.powerEngine()->sensorProjections().copyValues(_x.powerEngine()->sensorProjections());
  }

  void WavefieldValues::setValues(const Observations& obs)
  {
    Complex ell;
    /*ell.setUnitExp(obs.xiX);
    _x.setPowerX(ell);
    _x.setXiSlopeX(ell);
    _x.setXiConcavityX();
    _x.setKConcavity();*/

    ell.setUnitExp(obs.xiH);
    _h.setPowerH(ell);
    _h.setXiSlopeH(ell);
    _h.setXiConcavityH(ell);
    _h.setKConcavity();

    ell.setUnitExp(obs.xiZ);
    _z.setPowerZ(ell);
    _z.setXiSlopeZ(ell);
    _z.setXiConcavityZ(ell);
    _z.setKConcavity();
  }

  double WavefieldValues::misfit(const Observations& obs)
  {
    double m=0;
    //m+=_x.misfit(obs.powerX, obs.invObsPowerX, obs.kConcavityX, obs.invObsKConcavityX);
    m+=_h.misfit(obs.powerH, obs.invObsPowerH, obs.kConcavityH, obs.invObsKConcavityH);
    m+=_z.misfit(obs.powerZ, obs.invObsPowerZ, obs.kConcavityZ, obs.invObsKConcavityZ);
    return m;
  }

  QString WavefieldValues::detailedMisfit(const Observations& obs)
  {
    QString s;
    s+=QString(" ===> %1\n").arg(misfit(obs));
    s+=_x.detailedMisfit('X');
    s+=_h.detailedMisfit('H');
    s+=_z.detailedMisfit('Z');
    return s;
  }

  void WavefieldValues::setCrossSpectrumDerivative(const ComplexMatrix& df)
  {
    _dfInv=*_x.powerEngine()->crossSpectrum();
    _dfInv*=df;
    _dfInv*=*_x.powerEngine()->crossSpectrum();
    _dfInv=-_dfInv;
    _x.setDerivativeCrossSpectrum(&_dfInv);
    _h.setDerivativeCrossSpectrum(&_dfInv);
    _z.setDerivativeCrossSpectrum(&_dfInv);
  }

  void WavefieldValues::setDerivativeValues()
  {
    /*_x.setPowerDerivative();
    _x.setKSlopeDerivative();
    _x.setKConcavityDerivative();
    _x.setXiSlopeDerivative();*/

    _h.setPowerDerivative();
    _h.setKSlopeDerivative();
    _h.setKConcavityDerivative();
    _h.setXiSlopeDerivative();

    _z.setPowerDerivative();
    _z.setKSlopeDerivative();
    _z.setKConcavityDerivative();
    _z.setXiSlopeDerivative();
  }

  double WavefieldValues::misfitDerivative(const Observations& obs)
  {
    return /*_x.misfitDerivative(obs.invObsPowerX, obs.invObsKConcavityX)*/
          +_h.misfitDerivative(obs.invObsPowerH, obs.invObsKConcavityH)
          +_z.misfitDerivative(obs.invObsPowerZ, obs.invObsKConcavityZ);
  }

} // namespace ArrayCore

