/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-10-25
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SensorOrientationTaskManager.h"
#include "SensorOrientationWorker.h"
#include "SensorOrientationFunction.h"
#include "CrossSpectrumSplit.h"

namespace ArrayCore {

  /*!
    \class SensorOrientationTaskManager SensorOrientationTaskManager.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  SensorOrientationTaskManager::SensorOrientationTaskManager(const ArraySelection * array)
    : FKTaskManager(array)
  {
    TRACE;
    _observations=nullptr;
    _selectedStations=nullptr;
  }

  /*!
    Description of destructor still missing
  */
  SensorOrientationTaskManager::~SensorOrientationTaskManager()
  {
    TRACE;
    int n=parameters()->frequencySampling().count();
    for(int i=0; i<n; i++) {
      qDeleteAll(_observations[i]);
    }
    delete [] _observations;
  }

  bool SensorOrientationTaskManager::setParameters(const ArrayParameters * param)
  {
    TRACE;
    int n=param->frequencySampling().count();
    _observations=new VectorList<CrossSpectrumSplit *>[n];
    if(FKTaskManager::setParameters(param)) {
      selectStations();
      return true;
    } else {
      return false;
    }
  }

  LoopWorker * SensorOrientationTaskManager::newWorker()
  {
    TRACE;
    SensorOrientationWorker * w=new SensorOrientationWorker(&_array, parameters());
    w->setTaskManager(this);
    w->setOrientationFunction(_gridCache);
    if(!w->setGrid(_gridCache)) {
      delete w;
      w=nullptr;
    }
    return w;
  }

  void SensorOrientationTaskManager::selectStations()
  {
    int n=_array.count();
    _selectedStations=new bool[n];
    for(int i=0; i<n; i++) {
      _selectedStations[i]=false;
    }
    if(parameters()->sensorOrientationApertureAngle()>0.0) {
      for(int i=0; i<n; i++) {
        const Point2D& p=_array.relativePos(i);
        double az=p.azimuth();
        double d0=p.length();
        // If looking outwards (with the center in the back), if there any station from -pi/2 to pi/2?
        for(int j=0; j<n; j++) {
          if(i!=j) {
            const Point2D& q=_array.relativePos(j);
            if(q.length()>d0) {
              if(fabs(Angle::differenceRadians(p.azimuthTo(q),az))<
                 parameters()->sensorOrientationApertureAngle()/2.0) {
                _selectedStations[i]=true;
                break;
              }
            }
          }
        }
      }
    }
    if(parameters()->sensorOrientationMaximumRadius()>0.0) {
      for(int i=0; i<n; i++) {
        const Point2D& p=_array.relativePos(i);
        if(p.length()<parameters()->sensorOrientationMaximumRadius()) {
          _selectedStations[i]=true;
        } else {
          _selectedStations[i]=false;
        }
      }
    }
    // Is any selected? If none is selected, select all of them
    int i;
    for(i=0; i<n; i++) {
      if(_selectedStations[i]) {
        break;
      }
    }
    if(i==n) {
      for(int i=0; i<n; i++) {
        _selectedStations[i]=true;
      }
    }

    for(int i=0; i<n; i++) {
      App::log(tr("Station[%1] '%2': %3\n")
               .arg(i)
               .arg(_array.name(i))
               .arg(_selectedStations[i] ? tr("selected") : ""));
    }
  }

  void SensorOrientationTaskManager::addObservation(int frequencyIndex, CrossSpectrumSplit * split)
  {
    _observationMutex.lock();
    _observations[frequencyIndex].append(split);
    _observationMutex.unlock();
  }

  void SensorOrientationTaskManager::addResult(double frequency, const Vector<double>& position)
  {
    DoubleMatrix r(position.count()+1, 1);
    PrivateVector<double> t(position);
    t*=180.0/M_PI;
    while(t[0]>90.0) {
      t[0]-=180.0;
    }
    while(t[0]<-90.0) {
      t[0]+=180.0;
    }
    App::log(0, tr("New result at %1 Hz: %2\n")
             .arg(frequency, 5, frequency, 2, QChar(' '))
             .arg(t.toUserString()));
    r.at(0, 0)=frequency;
    double& ell=r.at(1, 0);
    ell=t.at(0);
    while(ell>90.0) {
      ell-=180.0;
    }
    while(ell<-90.0) {
      ell+=180.0;
    }
    for(int i=position.count()-1; i>0; i--) {
      r.at(i+1, 0)=t.at(i);
    }
    _resultMutex.lock();
    _results.append(r);
    _resultMutex.unlock();
  }

  /*!
  */
  bool SensorOrientationTaskManager::save(QString fileName, const QString& log)
  {
    TRACE;
    Q_UNUSED(fileName);
    Q_UNUSED(log);
    TODO_WARNING;
#ifdef ONE_INVERSION_PER_FREQUENCY
    if(!fileName.endsWith("-ell.max")) {
      if(fileName.endsWith(".max")) {
        fileName.chop(4);
      }
      fileName+="-ell.max";
    }
    QFile f(fileName);
    if(f.open(QIODevice::WriteOnly)) {
      //std::sort(_results.begin(), _results.end());
      QTextStream s(&f);
      s.setRealNumberPrecision(20);
      s << "# MAX FORMAT RELEASE 1.1\n";
      CoreApplication::signature(s);
      s << "#\n"
           "# BEGIN STATION LIST\n"
           "#\n";
      s << _array.toString(ArraySelection::Absolute);
      s << "#\n"
           "# END STATION LIST\n"
           "#\n";
      s << "#\n"
           "# BEGIN PARAMETERS\n"
           "#\n";
      s << parameters()->AbstractParameters::toString();
      s << parameters()->toString();
      s << "#\n"
           "# END PARAMETERS\n"
           "#\n";
      s << "#\n"
           "# BEGIN LOG\n"
           "#\n";
      s << log;
      s << "#\n"
           "# END LOG\n"
             "#\n";
      s << "#\n"
           "# BEGIN DATA\n"
           "#\n";
      s << "# frequency ellipticity correction1 correction2...\n";
      for(VectorList<DoubleMatrix>::iterator it=_results.begin(); it!=_results.end(); it++) {
        const DoubleMatrix& m=*it;
        int n=m.rowCount();
        for(int i=0; i<n; i++) {
          s << m.at(i, 0) << " ";
        }
        s << "\n";
      }
    } else {
      App::log(tr("Error saving results to '%1'.\n").arg(fileName));
      return false;
    }
    App::log(tr("Results saved in '%1'.\n").arg(fileName));
#endif
    return true;
  }

} // namespace ArrayCore

