/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2020-01-28
**  Copyright: 2020
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKRayleighFixedEll.h"
#include "FKCrossSpectrum.h"
#include "FKPower.h"

namespace ArrayCore {

  /*!
    \class HRFKRayleighFixedEll HRFKRayleighFixedEll.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  bool HRFKRayleighFixedEll::setFixedEllipticity(double frequency)
  {
    if(_fixedEllipticities.isEmpty()) {
      return true;
    } else {
      if(_fixedEllipticities.isInsideRange(frequency)) {
        _currentFixedEllipticity=_fixedEllipticities.interpole(frequency, LogScale, LinearScale).y();
        return true;
      } else {
        App::log(tr("  Frequency %1 Hz out of fixed ellipticity range\n").arg(frequency));
      }
    }
    return false;
  }

  double HRFKRayleighFixedEll::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
      steering()->oneComponent(index, powerEngine());
      const Vector<Complex>& e=steering()->twoComponentRayleighRadial(_currentFixedEllipticity,
                                                                      powerEngine());
      // Because the elipticity is fixed the type of power (h, z, s or x) does not
      // matter.
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(index));
      powerEngine()->setHighResolutionValue(e);
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }

  double HRFKRayleighFixedEll::value(Vector<double>& k) const
  {
    if(isInsideLimits(k)) {
      steering()->oneComponent(k, powerEngine());
      const Vector<Complex>& e=steering()->twoComponentRayleighRadial(_currentFixedEllipticity,
                                                                      powerEngine());
      // Because the elipticity is fixed the type of power (h, z, s or x) does not
      // matter.
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(k));
      powerEngine()->setHighResolutionValue(e);
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }

  /*!
    Load fixed ellipticities from \a fileName.
    They are in radians.
  */
  bool HRFKRayleighFixedEll::loadFixedEllipticityCurve(const QString& fileName)
  {
    TRACE;
    QFile f(fileName);
    if(f.open(QIODevice::ReadOnly)) {
      QTextStream s(&f);
      LineParser lp;
      _fixedEllipticities.clear();
      bool ok=true;
      int lineNumber=0;
      while(!s.atEnd()) {
        QString buf=s.readLine();
        lineNumber++;
        if(!buf.isEmpty() && buf[0]!='#') {
          lp.setString(buf);
          // Ellipticity is stored as e not xi.
          Point2D p(lp.toDouble(0, ok), tan(Angle::degreesToRadians(lp.toDouble(1, ok))));
          if(!ok) {
            App::log(tr("Error parsing line %1 in file '%2'\n").arg(lineNumber).arg(fileName));
            return false;
          }
          _fixedEllipticities.append(p);
        }
      }
      App::log(tr("%1 fixed ellipticities from %2 to %3 Hz\n")
               .arg(_fixedEllipticities.count())
               .arg(_fixedEllipticities.constAt(0).x())
               .arg(_fixedEllipticities.constAt(_fixedEllipticities.count()-1).x()));
      return true;
    } else {
      App::log(tr("Cannot open file '%1'\n").arg(fileName));
      return false;
    }
  }

} // namespace ArrayCore

