/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-08-26
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKRayleighAll.h"
#include "FKCrossSpectrum.h"
#include "FKPower.h"
#include "HREllipticityEngine.h"

namespace ArrayCore {

  /*!
    \class HRFKRayleighAll HRFKRayleighAll.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  void HRFKRayleighAll::initialize(const FKParameters& param)
  {
    HRFKRayleigh::initialize(param);
    _ellEngine=new HREllipticityEngine(cache(), param);
  }

  inline void HRFKRayleighAll::commitEvaluationStat() const
  {
#ifdef NEWTON_EVALUATION_STATS
    evaluationMutex.lock();
    if(_xi.evaluationCount>=20) {
      nEvaluations[20]++;
    } else {
      nEvaluations[_xi.evaluationCount]++;
    }
    evaluationMutex.unlock();
#endif
  }

  inline void HRFKRayleighAll::maximumEllipticity() const
  {
    // We admit values ouside -pi/2 to p/2 to avoid been trapped right at +/- pi/2
    // This happens if _xi is between the limits and the minimum
    _ellOptimization.maximize(_xi, this, M_PI/8.0);
    commitEvaluationStat();
  }

  double HRFKRayleighAll::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
      oneComponentSteering(index);
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(index));
      maximumEllipticity();
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  double HRFKRayleighAll::value(Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
      oneComponentSteering(kell);
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));
      maximumEllipticity();
      canonicalEllipticity(_xi.x);
      kell[2]=_xi.x;
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  double HRFKRayleighAll::valueFixed(const Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
      oneComponentSteering(kell);
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));
      _xi.x=kell[2];
      setFunctionValue(_xi);
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  void HRFKRayleighAll::setFunctionValue(Attributes& a) const
  {
    a.ell.setUnitExp(a.x);
    const Vector<Complex>& e=FKSteering::twoComponentRayleighCross(a.ell, powerEngine());
    powerEngine()->setHighResolutionValue(e);
    a.value=powerEngine()->value();
  }

  void HRFKRayleighAll::setFunctionDerivatives(Attributes& a) const
  {
    double value2=a.value*a.value;
    // Get first derivative
    const Vector<Complex>& de=FKSteering::twoComponentRayleighCrossGradient(a.ell, powerEngine());
    a.slope=-2.0*value2*powerEngine()->ellipticityGradient(de);
    // Get second derivative
    a.concavity=powerEngine()->xiSecondDerivative();
    a.concavity=2.0*(a.slope*a.slope/a.value+a.value-value2*a.concavity);
  }

  /*!
    Calculates a noise factor from the second derivative of beam power
    and the second derivative of the beam pattern.
  */
  double HRFKRayleighAll::wavenumberNormalizedConcavity(const Vector<double>& kell) const
  {
    oneComponentSteering(kell);
    powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));

    Complex ell;
    ell.setUnitExp(kell[2]);
    const Vector<Complex>& e0=FKSteering::twoComponentRayleighCross(ell, powerEngine());
    powerEngine()->setHighResolutionValue(e0);
    double p=powerEngine()->value();
    steering()->twoComponentSensorRadialProjections(kell, powerEngine());
    powerEngine()->highResolutionKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
    return powerEngine()->highResolutionKRadialSecondDerivative()/p;
  }

  /*!
    Analyse with

    cat /tmp/kderivatives | awk '{print $1, $2}' | gpcurve -derivative | gpcurve -derivative | figue -c
    cat /tmp/kderivatives | figue -c -y-columns 1,2
  */
  void HRFKRayleighAll::testKDerivatives(const Vector<double>& kell) const
  {
    powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));
    Complex ell;
    ell.setUnitExp(kell[2]);

    QFile f("/tmp/kderivatives");
    f.open(QIODevice::WriteOnly);
    QTextStream s(&f);

    PrivateVector<double> k1(kell);
    k1/=k1.length(0, 1);
    for(double k=0; k<0.5; k+=0.0001) {
      PrivateVector<double> kv(k1);
      kv*=k;
      oneComponentSteering(kv);
      const Vector<Complex>& e0=FKSteering::twoComponentRayleighCross(ell, powerEngine());
      powerEngine()->setHighResolutionValue(e0);
      double p=powerEngine()->value();
      steering()->twoComponentSensorRadialProjections(kv, powerEngine());
      double dp=powerEngine()->highResolutionKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
      double ddp=powerEngine()->highResolutionKRadialSecondDerivative();
      s << k << " " << p*1e16 << " " << dp*1e16 << " " << ddp*1e16 << Qt::endl;
    }
    f.close();
    exit(0);
  }

} // namespace ArrayCore

