/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-06-28
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKDirectRayleighVertical.h"

namespace ArrayCore {

  /*!
    \class HRFKDirectRayleighVertical HRFKDirectRayleighVertical.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

// Get a detailled timing report of the processing
// Run with verbosity level to 3 to get the report.
// Configure with '-D PROCESS_TIMING'
#ifdef PROCESS_TIMING
  static double tref=0.0;
  static Mutex globalTimeMutex;
#endif

  HRFKDirectRayleighVertical::HRFKDirectRayleighVertical(FKCache * cache)
    : HRFKDirectRayleigh(cache)
  {
    // We cannot start iterating from 0 because it is a singular point
    _xi.x=0.5*M_PI;
  }

  HRFKDirectRayleighVertical::~HRFKDirectRayleighVertical()
  {
#ifdef PROCESS_TIMING
    App::log(3, tr("HRFKDirectRayleighVertical: %1 (ms; refine)\n")
              .arg(tref*1e-6));
#endif
  }

  inline void HRFKDirectRayleighVertical::maximumEllipticity() const
  {
    double xi0=_xi.x;
    if(!_ellOptimization.maximize(_xi, this, M_PI/8.0)) {
      App::log(tr("Error maximizing beampower for ellipticity (vertical)\n"));
      // to debug optimization errors
      _ellOptimization.curves<HRFKDirectRayleighVertical, Attributes>(this, -M_PI/2, M_PI/2, M_PI/1800.0);
      _xi.x=xi0;
      _ellOptimization.maximize(_xi, this, M_PI/8.0);
    }
  }

  double HRFKDirectRayleighVertical::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
      steering()->threeComponentRayleigh(index, powerEngine());
      maximumEllipticity();
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  double HRFKDirectRayleighVertical::value(Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
#ifdef PROCESS_TIMING
      QElapsedTimer chrono;
      chrono.start();
#endif
      steering()->threeComponentRayleigh(kell, powerEngine());
      maximumEllipticity();
      canonicalEllipticity(_xi.x);
      kell[2]=_xi.x;
#ifdef PROCESS_TIMING
      qint64 t=chrono.nsecsElapsed();
      globalTimeMutex.lock();
      tref+=t;
      globalTimeMutex.unlock();
#endif
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  void HRFKDirectRayleighVertical::setFunctionValue(Attributes& a) const
  {
    a.ell.setUnitExp(a.x);
    const Vector<Complex>& e=FKSteering::threeComponentRayleighVertical(a.ell, powerEngine());
    powerEngine()->setHighResolutionValue(e);
    a.value=powerEngine()->value();
  }

  void HRFKDirectRayleighVertical::setFunctionDerivatives(Attributes& a) const
  {
    double value2=a.value*a.value;
    // Get first derivative
    const Vector<Complex>& de=FKSteering::threeComponentRayleighVerticalGradient(a.ell, powerEngine());
    a.slope=-2.0*value2*powerEngine()->ellipticityGradient(de);
    // Get second derivative
    double cot=a.ell.re()/a.ell.im();
    a.concavity=powerEngine()->xiSecondDerivative();
    a.concavity=2.0*(a.slope*a.slope/a.value-cot*a.slope-value2*a.concavity);
  }

  /*!
    Calculates a noise factor from the second derivative of beam power
    and the second derivative of the beam pattern.
  */
  double HRFKDirectRayleighVertical::wavenumberNormalizedConcavity(const Vector<double>& kell) const
  {
    steering()->threeComponentRayleigh(kell, powerEngine());

    Complex ell;
    ell.setUnitExp(kell[2]);
    const Vector<Complex>& e0=FKSteering::threeComponentRayleighVertical(ell, powerEngine());
    powerEngine()->setHighResolutionValue(e0);
    double p=powerEngine()->value();
    steering()->threeComponentSensorRadialProjections(kell, powerEngine());
    powerEngine()->highResolutionKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
    return powerEngine()->highResolutionKRadialSecondDerivative()/p;
  }

  /*!
    value() or valueFixed() must be called before with the same \a kell.
    If \a axis is 0, radial wavenumber concavity is returned.
    If \a axis is 2, ellipticity concavity is returned.
  */
  double HRFKDirectRayleighVertical::concavity(const Vector<double>& kell, int axis) const
  {
    // Recompute fe, altered by setFunctionDerivatives()
    setFunctionValue(_xi);
    switch(axis) {
    case 2:
      setFunctionDerivatives(_xi); // Required if valueFixed() is called before
      return _xi.concavity;
    default:
      steering()->threeComponentSensorRadialProjections(kell, powerEngine());
      powerEngine()->highResolutionKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
      return powerEngine()->highResolutionKRadialSecondDerivative();
    }
  }

} // namespace ArrayCore

