/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-05-15
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKDirectRadial.h"
#include "FKCrossSpectrum.h"
#include "FKPower.h"

namespace ArrayCore {

  /*!
    \class HRFKDirectRadial HRFKDirectRadial.h
    \brief Radial high resolution FK power with direct steering of a 2-component cross spectrum

    Radial high resolution FK power versus wavenumber (k) function.
    The cross-spectrum is made of raw component E and N.
    Projection on the radial direction is included in the steering matrix.
  */

  void HRFKDirectRadial::initialize(const FKParameters& param)
  {
    setPowerEngine(new FKPower(cache()->array()->count(), 2));
    if(_crossSpectrum) {
      powerEngine()->setCrossSpectrum(_crossSpectrum->matrix());
    }
    steeringInitialization()->setSensorPositionErrors(param);
    steeringInitialization()->setSensorOrientationErrors(param);
    steeringInitialization()->setPhaseShifts(param.gridStep(), param.effectiveGridSize());
    steeringInitialization()->setRadialRotations(param.gridStep());
  }

  void HRFKDirectRadial::addCrossSpectrum()
  {
    _crossSpectrum->addHorizontal();
  }

  void HRFKDirectRadial::addIncoherentNoise(double horizontal, double )
  {
    ComplexMatrix& covmat=*_crossSpectrum->matrix();
    int n=covmat.rowCount();
    for(int i=0; i<n; i++) {
      covmat.at(i, i)+=horizontal;
    }
  }

  double HRFKDirectRadial::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
      const Vector<Complex>& e=steering()->twoComponentHorizontal(index, powerEngine());
      powerEngine()->setHighResolutionValue(e);
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }

  double HRFKDirectRadial::value(Vector<double>& k) const
  {
    if(isInsideLimits(k)) {
      const Vector<Complex>& e=steering()->twoComponentRadial(k, powerEngine());
      powerEngine()->setHighResolutionValue(e);
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }

} // namespace ArrayCore

