/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-02-14
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "FKTool.h"
#include "KmaxSolver.h"
#include "KminSolver.h"
#include "FKResults.h"
#include "FKTaskManager.h"
#include "SensorOrientationTaskManager.h"

namespace ArrayCore {

  /*!
    \class FKTool FKTool.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  FKTool::FKTool(QObject * parent)
    : AbstractArrayTool(parent)
  {
    TRACE;
    _kmaxSolver=nullptr;
    setResults(new FKResults);
    setParameters(new FKParameters);
  }

  /*!
    Description of destructor still missing
  */
  FKTool::~FKTool()
  {
    TRACE;
    delete _kmaxSolver;
  }

  bool FKTool::setSubPool(SubSignalPool *subPool)
  {
    TRACE;
    if(AbstractArrayTool::setSubPool(subPool)) {
      return true;
    } else {
      return false;
    }
  }

  bool FKTool::setLoop()
  {
    TRACE;
    if(!AbstractArrayTool::setLoop()) {
      return false;
    }

    selectStations();
    if(!setFrequencySampling()) {
      return false;
    }

    FKParameters * param=parameters();
    if(param->kmin()==0.0 ||
       param->kmax()==0.0) {
      setWaveNumberGrid();
    }

    switch(param->processType()) {
    case FKParameters::ARDS:
    case FKParameters::SensorOrientation:
    case FKParameters::RDSSingle:
    case FKParameters::RTBF:
    case FKParameters::ARTBF:
    case FKParameters::RTBFFixedEll:
      break;
    case FKParameters::Undefined:
      ASSERT(false);
    case FKParameters::Conventional:
    case FKParameters::ConventionalRadial:
    case FKParameters::ConventionalTransverse:
    case FKParameters::ConventionalRayleigh:
    case FKParameters::Capon:
    case FKParameters::CaponRadial:
    case FKParameters::CaponTransverse:
    case FKParameters::PoggiVertical:
    case FKParameters::PoggiRadial:
    case FKParameters::Omni:
    case FKParameters::LDS2:
    case FKParameters::LDS3:
    case FKParameters::ActiveConventional:
    case FKParameters::ActiveCapon:
    case FKParameters::ActiveConventionalRayleigh:
    case FKParameters::ActiveRTBF:
      // Make sure that the user did it correctly. There is no ellipticity
      // computed, hence no result are produced if this option is off.
      param->setSaveUndefinedEllipticities(true);
      break;
    }
    // At this step parameters are fine and they will be no longer modified
    return AbstractArrayTool::setLoop();
  }

  ParallelTaskManager * FKTool::createLoop()
  {
    switch(parameters()->processType()) {
    case FKParameters::SensorOrientation:
      return new SensorOrientationTaskManager(array());
    default:
      break;
    }
    return new FKTaskManager(array());
  }

  /*!
    Called automatically by setParameters if minimum and maximum
    waveNumber are not set.
  */
  bool FKTool::setWaveNumberGrid()
  {
    TRACE;
    if(_kmaxSolver) {
      _kmaxSolver->terminate();
      _kmaxSolver->deleteLater();
      _kmaxSolver=nullptr;
    }

    VectorList<Point2D> stations;
    int n=array()->count();
    for(int i=0; i<n; i++) {
      stations.append(Point2D(array()->relativePos(i)));
    }
    // Estimate kmin and kmax from station coordinates
    KminSolver kminSolver(stations);
    bool ok=true;
    double kmin=kminSolver.calculate(ok);
    if (ok && kmin>0.0) {
      App::log(tr("Computed kmin=%1 rad/m\n").arg(kmin));
      parameters()->setKmin(kmin);
      _kmaxSolver=new KmaxSolver(stations, kmin, 0.25);
      connect(_kmaxSolver, SIGNAL(finished()), this, SLOT(setComputedKmax()));
      if(!_kmaxSolver->calculate()) {
        setComputedKmax(_kmaxSolver); // For 1D array, calculate() returns immediately
      }
    } else {
      App::log(tr("Error computing kmin for %1 station coordinates:\n").arg(n));
      for(int i=0; i<n; i++) {
        App::log("  "+stations.at(i).toString());
      }
      return false;
    }
    return true;
  }

  void FKTool::setComputedKmax(KmaxSolver * solver)
  {
    TRACE;
    // Upon termination, _kmaxSolver might be replaced by another solver
    // and this function called by an old event still in the stack
    // Make sure that it still refers to the current solver.
    if(!solver) {
      solver=qobject_cast<KmaxSolver *>(sender());
    }
    if(solver && _kmaxSolver==solver) {
      bool ok;
      double kmax=_kmaxSolver->kmax(ok);
      if(ok) {
        App::log(tr("Computed kmax=%1 rad/m\n").arg(kmax) );
        parameters()->setKmax(kmax);
        emit waveNumberGridReady();
      } else {
        App::log(tr("Error computing kmax for station coordinates\n") );
      }
      _kmaxSolver->deleteLater();
      _kmaxSolver=nullptr;
    }
  }  

  void FKTool::start(int threadCount, bool forceParallel)
  {
    TRACE;
    waitFinished(0);
    AbstractArrayTool::start(threadCount, forceParallel);
  }

  void FKTool::waitFinished(int ms)
  {
    TRACE;
    if(_kmaxSolver) {
      App::log(tr("Waiting for Kmax computation...\n") );
      _kmaxSolver->wait();
    }
    // Run loop to delete kmax solver
    CoreApplication::processEvents();
    AbstractArrayTool::waitFinished(ms);
  }

  bool FKTool::saveResults(const QString& fileName)
  {
    switch(parameters()->processType()) {
    case FKParameters::SensorOrientation:
      return results()->save(fileName, log()) && static_cast<SensorOrientationTaskManager *>(loop())->save(fileName, log());
    default:
      break;
    }
    return results()->save(fileName, log());
  }

} // namespace ArrayCore

