/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-04-04
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "FKSteering.h"
#include "FKParameters.h"

namespace ArrayCore {

  /*!
    \class FKSteering FKSteering.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing

    FKSteering is allocation in FKCache with new[], hence
    only default constructor is allowed.
  */
  FKSteering::FKSteering(const ArraySelection * array)
    : _coordinates(array->count(), 0.0),
      _xCoordinates(array->count(), 0.0),
      _yCoordinates(array->count(), 0.0)
  {
    _array=array;
    _kCount=0;
    _ellCount=0;
#ifdef FKSTEERING_1D
    _phaseShiftsX=nullptr;
    _phaseShiftsY=nullptr;
#else
    _phaseShifts=nullptr;
#endif
    _rotations=nullptr;
    _sensorOrientationErrors=nullptr;
    _sensorPositionErrors=false;

    // Store coordinates in various ways that can be helpful to speed up their access.
    for(int i=array->count()-1; i>=0; i--) {
      const Point2D& r=_array->relativePos(i);
      _coordinates[i]=Complex(r.x(), r.y());
      _xCoordinates[i]=r.x();
      _yCoordinates[i]=r.y();
    }
  }

  /*!
    Description of destructor still missing
  */
  FKSteering::~FKSteering()
  {
#ifdef FKSTEERING_1D
    if(_phaseShiftsX) {
      int halfCount=(_kCount-1)/2;
      int stationCount=_array->count();
      delete [] (_phaseShiftsX-halfCount*stationCount);
      delete [] (_phaseShiftsY-halfCount*stationCount);
    }
#else
    if(_phaseShifts) {
      int halfCount=(_kCount-1)/2;
      int stationCount=_array->count();
      delete [] (_phaseShifts-halfCount*(_kCount+1)*stationCount);
    }
#endif
    if(_rotations) {
      int halfCount=(_kCount-1)/2;
      delete [] (_rotations-halfCount*(_kCount+1));
    }
    delete [] _sensorOrientationErrors;
  }

#ifdef FKSTEERING_1D
  void FKSteering::setPhaseShifts(double step, double size)
  {
    if(!_phaseShiftsX) {
      int halfCount=qRound(size/step);
      _kCount=(2*halfCount+1);
      int stationCount=_array->count();
      int nx=_kCount*stationCount;
      _phaseShiftsX=new Complex[nx]+halfCount*stationCount;
      _phaseShiftsY=new Complex[nx]+halfCount*stationCount;

      App::log(tr("Pre-computing steering vectors for X and Y (%2 cells per axis): %1 Mb\n")
               .arg(2*nx*sizeof(Complex)/(1024.0*1024.0))
               .arg(_kCount));
      Complex * kx, * ky, x;
      double k;
      step=-step; // Avoid repeated operator-()
      for(int ik=-halfCount; ik<=halfCount; ik++) {
        k=ik*step;
        kx=_phaseShiftsX+ik*stationCount;
        ky=_phaseShiftsY+ik*stationCount;
        for(int i=0; i<stationCount; i++) {
          const Point2D& r=_array->relativePos(i);
          kx[i].setUnitExp(r.x()*k);
          ky[i].setUnitExp(r.y()*k);
        }
      }
      // Test
      /*Complex e, te;
      for(int ix=-halfCount; ix<=halfCount; ix++) {
        for(int iy=-halfCount; iy<=halfCount; iy++) {
          Complex * kx=_phaseShiftsX+ix*stationCount;
          Complex * ky=_phaseShiftsY+iy*stationCount;
          Point2D k(ix*step, iy*step);
          for(int i=0; i<stationCount; i++) {
            e=kx[i];
            e*=ky[i];
            const Point2D& r=_array->relativePos(i);
            te.setUnitExp(r.scalarProduct(k));
            e-=te;
            if(fabs(e.re())>1e-14 || fabs(e.im())>1e-14) {
              App::log(tr("Error at %1 %2: %3\n")
                        .arg(ix)
                        .arg(iy)
                        .arg((e).toString('g', 20)));
              exit(2);
            }
          }
        }
      }*/
    }
  }
#else
  void FKSteering::setPhaseShifts(double step, double size)
  {
    if(!_phaseShifts) {
      int halfCount=qRound(size/step);
      _kCount=(2*halfCount+1);
      int stationCount=_array->count();
      int nx=_kCount*stationCount;
      _phaseShifts=new Complex[_kCount*nx]+halfCount*(nx+stationCount);

      App::log(tr("Pre-computing steering vectors on the grid (%2 cells per axis): %1 Mb\n")
               .arg(_kCount*nx*sizeof(Complex)/(1024.0*1024.0))
               .arg(_kCount));

      Complex * valuesX, * values;
      Complex x;
      step=-step; // Avoid repeated operator-()
      for(int ky=-halfCount; ky<=halfCount; ky++) {
        valuesX=_phaseShifts+ky*nx;
        for(int kx=-halfCount; kx<=halfCount; kx++) {
          values=valuesX+kx*stationCount;
#if !defined COMPATIBILITY_3_4_1 && defined(__SSE3__) && !defined(Q_OS_DARWIN)
          // A faster method to compute scalar product using vectorization.
          // Tests using __builtin_ia32_mulpd showed that it is slightly slower than
          // using the complex multiplication.
          // Eventually with FMA4 extension, the full scalar product can be done
          // in a single instruction: not checked.
          Complex k(kx*step, -ky*step);
          for(int i=0; i<stationCount; i++) {
            x=k;
            x*=_coordinates[i];
            values[i].setUnitExp(x.re());
          }
#else
          // No impact on the results for vertical Capon
#  ifdef COMPATIBILITY_3_4_1
          double min=qCeil(size/step)*step;
          Point2D k(min+(kx+halfCount)*step, min+(ky+halfCount)*step);
#  else
          Point2D k(kx*step, ky*step);
#  endif
          for(int i=0; i<stationCount; i++) {
            const Point2D& r=_array->relativePos(i);
            values[i].setUnitExp(r.scalarProduct(k));
          }
#endif
        }
      }
    }
  }
#endif

  /*!
    Phase shifts must be initialized first.
  */
  void FKSteering::setRadialRotations(double step)
  {
    if(!_rotations) {
      ASSERT(_kCount>0);
      int halfCount=(_kCount-1)/2;
      _rotations=new Complex[_kCount*_kCount]+halfCount*(_kCount+1);

      App::log(tr("Pre-computing radial rotations on the grid (%2 cells per axis): %1 Mb\n")
               .arg(_kCount*_kCount*sizeof(Complex)/(1024.0*1024.0))
               .arg(_kCount));
      Complex * valuesX;

      for(int ky=-halfCount; ky<=halfCount; ky++) {
        valuesX=_rotations+ky*_kCount;
        for(int kx=-halfCount; kx<=halfCount; kx++) {
          Complex& v=valuesX[kx];
          v.set(kx*step, ky*step);
          double absv=v.abs();
          if(absv>0.0) {
            v/=v.abs();
          } else { // Null wavenumber, arbitrarily select east direction
            v.set(1.0, 0.0);
          }
        }
      }
    }
  }

  void FKSteering::setTransverseRotations(double step)
  {
    if(!_rotations) {
      ASSERT(_kCount>0);
      int halfCount=(_kCount-1)/2;
      _rotations=new Complex[_kCount*_kCount]+halfCount*(_kCount+1);

      App::log(tr("Pre-computing transverse rotations on the grid (%2 cells per axis): %1 Mb\n")
               .arg(_kCount*_kCount*sizeof(Complex)/(1024.0*1024.0))
               .arg(_kCount));
      Complex * valuesX;

      for(int ky=-halfCount; ky<=halfCount; ky++) {
        valuesX=_rotations+ky*_kCount;
        for(int kx=-halfCount; kx<=halfCount; kx++) {
          Complex& v=valuesX[kx];
          v.set(-ky*step, kx*step);     // Not the same rotation as radial
          double absv=v.abs();
          if(absv>0.0) {
            v/=v.abs();
          } else { // Null wavenumber, arbitrarily select east direction
            v.set(1.0, 0.0);
          }
        }
      }
    }
  }

  /*!
    Introduces position errors
  */
  void FKSteering::setSensorPositionErrors(const FKParameters& param)
  {
    if(!_sensorPositionErrors) {
      int stationCount=_array->count();
      Point2D delta;
      delta=param.sensorPositionError("@ALL");
      if(delta!=Point2D()) {
        for(int i=0; i<stationCount; i++) {
          _coordinates[i]+=Complex(delta.x(), delta.y());
          _xCoordinates[i]+=delta.x();
          _yCoordinates[i]+=delta.y();
        }
        App::log(tr("Sensor position error for all stations: %1\n").arg(delta.toString('g', 20)));
      }
      int availableErrorCount=param.sensorPositionErrorCount();
      delta=param.sensorPositionError("@RANDOM");
      if(delta!=Point2D()) {
        availableErrorCount--;
        // Output error values in a format auitable for FKParameter parser, to re-use these values as parameters
        Random r;
        QString str="Random sensor position errors:\n  ";
        for(int i=0; i<stationCount; i++) {
          // Make sure that the random delta is inside the ellipse
          Point2D randDelta;
          double invdx2=1.0/(delta.x()*delta.x());
          double invdy2=1.0/(delta.y()*delta.y());
          do {
            randDelta.setX(r.uniform(-delta.x(), delta.x()));
            randDelta.setY(r.uniform(-delta.y(), delta.y()));
          } while (randDelta.x()*randDelta.x()*invdx2>1.0-randDelta.y()*randDelta.y()*invdy2);
          _coordinates[i]+=Complex(randDelta.x(), randDelta.y());
          _xCoordinates[i]+=randDelta.x();
          _yCoordinates[i]+=randDelta.y();
          if(i>0) {
            str+=", \\\n  ";
          }
          str+=_array->name(i);
          str+=":";
          str+=randDelta.toString('g', 20);
        }
        App::log(str+"\n");
      }
      int usedErrorCount=0;
      for(int i=0; i<stationCount; i++) {
        QString name=_array->name(i);
        delta=param.sensorPositionError(name);
        if(delta!=Point2D()) {
          usedErrorCount++;
          _coordinates[i]+=Complex(delta.x(), delta.y());
          _xCoordinates[i]+=delta.x();
          _yCoordinates[i]+=delta.y();
          App::log(tr("Sensor position error for station '%1': %2\n").arg(name).arg(delta.toString('g', 20)));
        }
      }
      if(usedErrorCount<availableErrorCount) {
        App::log(tr("%1/%2 sensor position errors assigned to %3 stations\n")
                 .arg(usedErrorCount)
                 .arg(availableErrorCount)
                 .arg(stationCount));
      }
      _sensorPositionErrors=true;
    }
  }

  /*!
    Set internal vector with oritentation errors
  */
  void FKSteering::setSensorOrientationErrors(const FKParameters& param)
  {
    if(!_sensorOrientationErrors) {
      int stationCount=_array->count();
      _sensorOrientationErrors=new Complex[stationCount];
      // Set a default null error for all stations
      for(int i=0; i<stationCount; i++) {
        _sensorOrientationErrors[i]=Complex(1.0, 0.0);
      }
      int availableErrorCount=param.sensorOrientationErrorCount();
      double angle;
      angle=param.sensorOrientationError("@ALL");
      if(angle!=0.0) {
        availableErrorCount--;
        for(int i=0; i<stationCount; i++) {
          _sensorOrientationErrors[i].setUnitExp(Angle::degreesToRadians(angle));
        }
        App::log(tr("Sensor orientation error for all stations: %1\n").arg(angle));
      } else {
        angle=param.sensorOrientationError("@RANDOM");
        if(angle!=0.0) {
          availableErrorCount--;
          // Output error values in a format suitable for FKParameter parser, to re-use these values as parameters
          Random r;
          QString str="Random sensor orientation errors (deg):\n  ";
          for(int i=0; i<stationCount; i++) {
            double anglei=r.uniform(-angle, angle);
            _sensorOrientationErrors[i].setUnitExp(Angle::degreesToRadians(anglei));
            if(i>0) {
              str+=", \\\n  ";
            }
            str+=_array->name(i);
            str+=":";
            str+=QString::number(anglei, 'g', 20);
          }
          App::log(str+"\n");
        }
      }
      int usedErrorCount=0;
      for(int i=0; i<stationCount; i++) {
        QString name=_array->name(i);
        APP_LOG(1, tr("Orientation error for station '%1'?\n").arg(name));
        angle=param.sensorOrientationError(name);
        if(angle!=0.0) {
          usedErrorCount++;
          _sensorOrientationErrors[i].setUnitExp(Angle::degreesToRadians(angle));
          App::log(tr("Sensor orientation error for station '%1': %2\n").arg(name).arg(angle));
        }
      }
      if(usedErrorCount<availableErrorCount) {
        App::log(tr("%1/%2 sensor orientation errors assigned to %3 stations\n")
                 .arg(usedErrorCount)
                 .arg(availableErrorCount)
                 .arg(stationCount));
        APP_LOG(1, param.sensorOrientationErrorsToString()+"\n");
      }
    }
  }

  /*!
    \fn FKSteering::twoComponentHorizontal(const Vector<int>& index, FKPower * p) const

    Used with a wavenumber index for either radial or transverse projection.
    The type of projection is defined by calling either
      setRadialRotations(double step)
    or
      setTransverseRotations(double step)
  */

  /*!
    \fn FKSteering::twoComponentRadial(const Vector<double>& k, FKPower * p) const;

    Used with a plain wavenumber value for radial projection.
  */

  /*!
    \fn FKSteering::twoComponentTransverse(const Vector<double>& k, FKPower * p) const;

    Used with a plain wavenumber value for transverse projection.
  */

} // namespace ArrayCore

