/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-12-02
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef FKCROSSSPECTRUM_H
#define FKCROSSSPECTRUM_H

#include "ArrayCoreDLLExport.h"
#include "ArrayCrossSpectrum.h"
#include "AbstractFKFunction.h"
#include "ActiveFKSteering.h"

namespace ArrayCore {

  class ARRAYCORE_EXPORT FKCrossSpectrum : public ArrayCrossSpectrum
  {
  public:
    FKCrossSpectrum(const ArraySelection * array,
                    const FKParameters * param);
    ~FKCrossSpectrum();

    const FKParameters * parameters() const {return static_cast<const FKParameters *>(_parameters);}

    double rotationStep() const {return _rotateStep;}
    int rotationStepCount() const {return _rotateStepCount;}

    bool calculate(const VectorList<int>& blocks, AbstractFKFunction * function);

    virtual Point setActiveSource(const VectorList<int>& blocks, bool& ok);
    void forceActiveSource(const SeismicEvent * src) {_source=src;}

    const Point& sourceOffset() const {return _sourceOffset;}
    void setSourceOffset(const Point& o) {_sourceOffset=o;}

    void resize();
    void resetRotatedMatrices();

    virtual void addThreeComponent(const Complex * sensorRotations);
    // Test of optical fiber arrays
    //void addHorizontalRotatedRayleigh() {addHorizontalRotated(_rayleigh);}
    //void addHorizontalRotatedLove() {addHorizontalRotated(_love);}
    void addRotatedRadial(const Complex * sensorRotations);
    void addRotatedTransverse(const Complex * sensorRotations);
    void addRotatedRayleigh(const Complex * sensorRotations);
    virtual void addActiveVertical(const ActiveFKSteering& geom);
    virtual void addActiveRayleigh(const ActiveFKSteering& geom);
    virtual void addActiveTransverse(const ActiveFKSteering& geom);
    virtual void addActiveThreeComponent(const ActiveFKSteering& geom);
    void addRayleighSingle();

    void meanRotatedMatrices(int nBlocks);
    void meanRayleighSingle();

    bool invert();
    bool invertRotatedMatrices();
    bool invert(ComplexMatrix& covmat);

    inline const ComplexMatrix * rotatedMatrix(const Vector<double>& k) const;
    inline const ComplexMatrix * rotatedMatrix(const Vector<int>& index) const;
    inline ComplexMatrix * rotatedMatrix(int rotationIndex);

    inline const ComplexMatrix * rotatedRadialMatrix(const Vector<double>& k) const;
    const ComplexMatrix * verticalMatrix() const {return _verticalMatrix;}

    bool setMask();
  protected:
    bool selectStations(const Point& src);

    virtual void addRotatedRadial(int rotationIndex, const Complex * sensorRotations);
    virtual void addRotatedTransverse(int rotationIndex, const Complex * sensorRotations);
    virtual void addRotatedRayleigh(int rotationIndex, const Complex * sensorRotations);

    inline int rotationIndex(double kx, double ky) const;

    ComplexMatrix * _rotatedMatrices;
    ComplexMatrix * _rotatedRadialMatrices;
    ComplexMatrix * _verticalMatrix;
    ComplexMatrix * _horizontalMatrix;

    int _rotateStepCount;
    double _rotateStep, _invRotateStep;
  private:
    // Test of optical fiber arrays
    //void addHorizontalRotated(ComplexMatrix& covmat);

    bool setRayleighArrayMask();
    bool setOneComponentArrayMask();

    const SeismicEvent * _source;
    Point _sourceOffset;

    DoubleMatrix _arrayMask;
  };

  inline int FKCrossSpectrum::rotationIndex(double kx, double ky) const
  {
    double angle=::atan2(ky, kx); // From -M_PI to M_PI
    if(angle<0.0) {
      angle+=2.0*M_PI;
    }
    // Angle from 0 to M_PI
    int index=qRound(angle*_invRotateStep);
    if(index<_rotateStepCount) {
      return index;
    } else {
      return 0;
    }
  }

  inline const ComplexMatrix * FKCrossSpectrum::rotatedMatrix(const Vector<int>& index) const
  {
    return _rotatedMatrices+rotationIndex(index[0], index[1]);
  }

  inline const ComplexMatrix * FKCrossSpectrum::rotatedMatrix(const Vector<double>& k) const
  {
    return _rotatedMatrices+rotationIndex(k[0], k[1]);
  }

  inline ComplexMatrix * FKCrossSpectrum::rotatedMatrix(int rotationIndex)
  {
    return _rotatedMatrices+rotationIndex;
  }

  inline const ComplexMatrix * FKCrossSpectrum::rotatedRadialMatrix(const Vector<double>& k) const
  {
    return _rotatedRadialMatrices+rotationIndex(k[0], k[1]);
  }

} // namespace ArrayCore

#endif // FKCROSSSPECTRUM_H

