/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-05-08
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "FKCrossSpectrum.h"
#include "FKPower.h"
#include "ConvFKVertical.h"

namespace ArrayCore {

  /*!
    \class ConvFKVertical ConvFKVertical.h
    \brief Vertical conventional FK power

    Vertical conventional FK power versus wavenumber (k) function.
  */

  ConvFKVertical::ConvFKVertical(FKCache * cache)
    : AbstractFKFunction(cache)
  {
    cache->setPassiveSteering();
  }

  void ConvFKVertical::initialize(const FKParameters& param)
  {
    setPowerEngine(new FKPower(cache()->array()->count(), 1));
    if(_crossSpectrum) {
      powerEngine()->setCrossSpectrum(_crossSpectrum->matrix());
    }
    steeringInitialization()->setSensorPositionErrors(param);
    steeringInitialization()->setPhaseShifts(param.gridStep(), param.effectiveGridSize());
  }

  FunctionSearch * ConvFKVertical::createSearch(const FKParameters& param)
  {
    GridSearch2D * grid=new GridSearch2D;
    grid->setRelativePrecision(0, param.velocityRelativePrecision());
    grid->setRelativePrecision(1, param.velocityRelativePrecision());
    grid->setFunction(this);
    grid->setGrid(param.gridStep(), param.effectiveGridSize());
    return grid;
  }

  void ConvFKVertical::resetCrossSpectrum()
  {
    _crossSpectrum->resetMatrix();
  }

  void ConvFKVertical::addCrossSpectrum()
  {
    _crossSpectrum->addVertical();
  }

  void ConvFKVertical::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanMatrix(nBlocks);
  }

  void ConvFKVertical::addIncoherentNoise(double, double vertical)
  {
    int n=_crossSpectrum->array().count();
    ComplexMatrix& covmat=*_crossSpectrum->matrix();
    for(int i=0; i<n; i++) {
      covmat.at(i, i)+=vertical;
    }
  }

  double ConvFKVertical::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
      powerEngine()->setConventionalValue(oneComponentSteering(index));
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }

  double ConvFKVertical::value(Vector<double>& k) const
  {
    if(isInsideLimits(k)) {
      powerEngine()->setConventionalValue(oneComponentSteering(k));
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }

} // namespace ArrayCore

