/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-07-22
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ConvFKRayleighVertical.h"
#include "FKCrossSpectrum.h"

namespace ArrayCore {

  /*!
    \class ConvFKRayleighVertical ConvFKRayleighVertical.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  ConvFKRayleighVertical::ConvFKRayleighVertical(FKCache * cache)
    : ConvFKRayleigh(cache)
  {
    // We cannot start iterating from 0 because it is a singular point
    _xi.x=0.5*M_PI;
  }

  inline void ConvFKRayleighVertical::minimumEllipticity() const
  {
    _ellOptimization.minimize(_xi, this, M_PI/8.0);
  }

  double ConvFKRayleighVertical::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
      oneComponentSteering(index);
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(index));
      minimumEllipticity();
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  double ConvFKRayleighVertical::value(Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
      oneComponentSteering(kell);
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));
      minimumEllipticity();
      canonicalEllipticity(_xi.x);
      kell[2]=atan(-1.0/tan(_xi.x));
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  void ConvFKRayleighVertical::setFunctionValue(Attributes& a) const
  {
    a.ell.setUnitExp(a.x);
    const Vector<Complex>& e=FKSteering::twoComponentRayleighVertical(a.ell, powerEngine());
    powerEngine()->setConventionalValue(e);
    a.value=powerEngine()->value();
  }

  void ConvFKRayleighVertical::setFunctionDerivatives(Attributes& a) const
  {
    // Get first derivative
    const Vector<Complex>& de=FKSteering::twoComponentRayleighVerticalGradient(a.ell, powerEngine());
    a.slope=2.0*powerEngine()->factorN()*powerEngine()->ellipticityGradient(de);
    // Get second derivative
    double cot=a.ell.re()/a.ell.im();
    a.concavity=powerEngine()->factorN()*powerEngine()->xiSecondDerivative();
    a.concavity=2.0*(a.concavity-cot*a.slope);
  }

} // namespace ArrayCore

