/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-02-02
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "FKCrossSpectrum.h"
#include "FKPower.h"
#include "ConvFKRadial.h"

namespace ArrayCore {

  /*!
    \class ConvFKRadial ConvFKRadial.h
    \brief Radial conventional FK power

    Radial conventional FK power versus wavenumber (k) function.
    The cross-spectrum is already projected on the radial direction.
  */

  void ConvFKRadial::initialize(const FKParameters& param)
  {
    steeringInitialization()->setSensorOrientationErrors(param);
    ConvFKVertical::initialize(param);
  }

  void ConvFKRadial::resetCrossSpectrum()
  {
    _crossSpectrum->resetRotatedMatrices();
  }

  void ConvFKRadial::addCrossSpectrum()
  {
    _crossSpectrum->addRotatedRadial(steering()->sensorOrientationErrors());
  }

  void ConvFKRadial::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanRotatedMatrices(nBlocks);
  }

  void ConvFKRadial::addIncoherentNoise(double horizontal, double)
  {
    int n=_crossSpectrum->array().count();
    int rotCount=_crossSpectrum->rotationStepCount();
    for(int rotIndex=0; rotIndex<rotCount; rotIndex++) {
      ComplexMatrix& covmat=*_crossSpectrum->rotatedMatrix(rotIndex);
      for(int i=0; i<n; i++) {
        covmat.at(i, i)+=horizontal;
      }
    }
  }

  const ComplexMatrix * ConvFKRadial::crossSpectrum() const
  {
    if(_crossSpectrum) {
      return _crossSpectrum->rotatedMatrix(0);
    } else {
      return nullptr;
    }
  }

  double ConvFKRadial::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(index));
      powerEngine()->setConventionalValue(oneComponentSteering(index));
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }

  double ConvFKRadial::value(Vector<double>& k) const
  {
    if(isInsideLimits(k)) {
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(k));
      powerEngine()->setConventionalValue(oneComponentSteering(k));
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }

} // namespace ArrayCore

