/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-09-17
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ActiveConvFKVertical.h"
#include "FKCrossSpectrum.h"

namespace ArrayCore {

  /*!
    \class ActiveConvFKVertical ActiveConvFKVertical.h
    \brief Brief description of class still missing

    Full description of class still missing.

    Source may vary from one cross spectrum to the next. Useless to cache a grid.
  */

  ActiveConvFKVertical::ActiveConvFKVertical(FKCache * cache)
    : AbstractFKFunction(cache)
  {
    _steering=nullptr;
  }

  ActiveConvFKVertical::~ActiveConvFKVertical()
  {
    delete _steering;
  }

  void ActiveConvFKVertical::initialize(const FKParameters&)
  {
    // No power initialization here because the number
    // of selected station is not yet known
  }

  bool ActiveConvFKVertical::setSource(const VectorList<int>& blocks)
  {
    bool ok=true;
    // According to distance parameters, the number of selected stations
    // after this call may be less than the number of stations of the
    // array in cache which is the parent of the array in cross spectrum.
    Point src=_crossSpectrum->setActiveSource(blocks, ok);
    if(!ok) {
      return false;
    }
    // When source grid is defined
    src+=_crossSpectrum->sourceOffset();
    // Number of selected stations is now correct
    setPowerEngine(new FKPower(_crossSpectrum->array().count(), 1));
    powerEngine()->setCrossSpectrum(_crossSpectrum->matrix());
    // Cache cannot be modified like this: not thread safe
    // Number of receivers and their indexes may vary, so not wise to put it in the main cache
    // Steering must be kept by the function itself
    if(!_steering) {
      _steering=new ActiveFKSteering;
    }
    _steering->setArray(&_crossSpectrum->array());
    _steering->setSource(src);
    _steering->setDistances();
    _steering->setAmplitudeFactors();
    return true;
  }

  FunctionSearch * ActiveConvFKVertical::createSearch(const FKParameters& param)
  {
    GridSearch1D * grid=new GridSearch1D(2);
    grid->setRelativePrecision(0, param.velocityRelativePrecision());
    grid->setFunction(this);
    grid->setGrid(param.gridStep(), param.effectiveGridSize(), param.gridStep());
    return grid;
  }

  void ActiveConvFKVertical::resetCrossSpectrum()
  {
    _crossSpectrum->resetMatrix();
  }

  void ActiveConvFKVertical::addCrossSpectrum()
  {
    // Correction for geometrical attenuation can also be achieved in steering
    // However, the steering correction is not the same for Conventional and High Resolution
    // Hence, simpler to correct the cross-spectrum.
    _crossSpectrum->addActiveVertical(*_steering);
  }

  void ActiveConvFKVertical::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanMatrix(nBlocks);
  }

  void ActiveConvFKVertical::addIncoherentNoise(double, double vertical)
  {
    int n=_crossSpectrum->array().count();
    ComplexMatrix& covmat=*_crossSpectrum->matrix();
    for(int i=0; i<n; i++) {
      covmat.at(i, i)+=vertical;
    }
  }

  double ActiveConvFKVertical::value(Vector<double>& k) const
  {
    k[1]=0.0; // Only k[0] is modified by GridSearch
    if(isInsideLimits(k)) {
      const Vector<Complex>& e=_steering->oneComponent(k[0], powerEngine());
      powerEngine()->setConventionalValue(e);
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }

} // namespace ArrayCore

