/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-07-22
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef ABSTRACTFKRAYLEIGH_H
#define ABSTRACTFKRAYLEIGH_H

#include "AbstractFKFunction.h"
#include "ArrayCoreDLLExport.h"

namespace ArrayCore {

  class AbstractEllipticityEngine;

  class ARRAYCORE_EXPORT AbstractFKRayleigh : public AbstractFKFunction
  {
  public:
    AbstractFKRayleigh(FKCache * cache);
    ~AbstractFKRayleigh();

    void initialize(const FKParameters& param);

    struct Attributes {
      double x, value, slope, concavity;
      int evaluationCount;
      Complex ell;
    };

    void setAssociatedResults(const Vector<double>& kell, double power,
                              FKPeaks::Value& val) const;
    double radialPower(Vector<double>& kell) const;
    double radialConcavity(Vector<double>& kell, int axis) const;
    double verticalPower(Vector<double>& kell) const;
    double verticalConcavity(Vector<double>& kell, int axis) const;

    static inline void canonicalEllipticity(double& xi);

    static PolynomialFraction phiH(double eh, double ez, double ch);
    static PolynomialFraction phiZ(double eh, double ez, double cz);
    static PolynomialFraction phiA(double eh, double ez, double ea, double ca);
    static PolynomialFraction cH(double eh, double ez);
    static PolynomialFraction cZ(double eh, double ez);
    static PolynomialFraction cA(double eh, double ez, double ea);
    static Polynomial phitH(double eh, double ez, double th);
    static PolynomialFraction phitZ(double eh, double ez, double tz);
    static PolynomialFraction phitA(double eh, double ez, double ea, double ta);
    static double minimumConcavityFactorHZ(double eh, double ez);
    static double minimumConcavityFactorA(double eh, double ez, double ea);
  protected:
    void rawEllipticity(const Vector<double>& kell, FKPeaks::Value& val) const;
    void threeComponentEllipticity(const Vector<double>& kell, FKPeaks::Value& val) const;

    NewtonOptimization1D _ellOptimization;
    mutable Attributes _xi;
    struct Solution {
      double ellipticity, verticalNoise, horizontalNoise;
    };
    AbstractEllipticityEngine * _ellEngine;
    mutable Solution _solution2;
    FKParameters::EllipticityCorrection _ellipticityCorrection;
  private:
    static void showRoots(const VectorList<double>& roots,
                          const PolynomialFraction& phi, const PolynomialFraction& phiDer,
                          double eh, double ez);
    static double bestRoot(const VectorList<double>& roots, const PolynomialFraction& phi, double phim);
    double solve(const PolynomialFraction& phi, int& rootCount,
                 const double& eh, const double& ez, double phim) const;
    struct PhiSolution {
      double xir;
      double value;
      int rootCount;
      int index;
    };
    static bool compareXir(const PhiSolution& s1, const PhiSolution& s2);
  };

  inline void AbstractFKRayleigh::canonicalEllipticity(double& xi)
  {
    while(xi<-M_PI/2.0) {
      xi+=M_PI;
    }
    while(xi>M_PI/2.0) {
      xi-=M_PI;
    }
  }

} // namespace ArrayCore

#endif // ABSTRACTFKRAYLEIGH_H

