/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <math.h>

#include "AbstractFKFunction.h"
#include "FKGradientSearch.h"
#include "FKCrossSpectrum.h"

#include "ConvFKVertical.h"
#include "ConvFKRadial.h"
#include "ConvFKTransverse.h"
#include "ConvFKRayleighAll.h"
#include "OmniFKVertical.h"
#include "OmniFKRayleigh.h"
#include "PoggiFKVertical.h"
#include "PoggiFKRadial.h"
#include "HRFKVertical.h"
#include "HRFKRadial.h"
#include "HRFKTransverse.h"
#include "HRFKRayleighSquared.h"
#include "HRFKRayleighAll.h"
#include "HRFKRayleighFixedEll.h"
#include "HRFKDirectRayleighAll.h"
#include "HRFKDirectLove.h"
#include "SensorOrientationCapon.h"
#include "HRFKDirectRadial.h"
#include "HRFKDirectTransverse.h"
#include "SingleDirectRayleigh.h"
#include "ActiveConvFKVertical.h"
#include "ActiveHRFKVertical.h"
#include "ActiveHRFKRayleigh.h"
#include "ActiveConvFKRayleighAll.h"

namespace ArrayCore {

  /*!
    \fn template <class Function, class Attributes> printEllipticityCurve(const Function * func) const
    Ouput power and its derivative to a temporary file.

    Numerical verification:
    cat /tmp/ellderivatives | awk '{print $1,$2}' | gpcurve -derivative | figue -c -l der.legend -e d1.layer
    cat /tmp/ellderivatives | awk '{print $1,$3}' | gpcurve -derivative | figue -c -l der.legend -e d2.layer
    cat /tmp/ellderivatives | figue -c -y-columns 1,2,3 -append-layer d1.layer -append-layer d2.layer&
  */

  /*!
    \class AbstractFKFunction AbstractFKFunction.h
    \brief AbstractFKFunction map calculator

    Compute the AbstractFKFunction map for a given array of stations.
  */

  /*!
    Init unlimited boundaries for wavenumber spaces (array, slowness and total)
    Null Gaussian weights.
  */
  AbstractFKFunction::AbstractFKFunction(FKCache * cache)
  {
    _powerEngine=nullptr;
    _crossSpectrum=nullptr;
    _cache=cache;

    _gridStep=0.0;
    _squaredKmaxArray=0.0;
    _squaredKminArray=0.0;
    _squaredSlowMin=0.0;
    _squaredSlowMax=std::numeric_limits<double>::infinity();
    _squaredKminTotal=0.0;
    _squaredKmaxTotal=std::numeric_limits<double>::infinity();
    _gridSquaredKminTotal=0.0;
    _gridSquaredKmaxTotal=std::numeric_limits<double>::infinity();
    _minAzimuth=0.0;
    _maxAzimuth=2.0*M_PI;
  }

  AbstractFKFunction::~AbstractFKFunction()
  {
    delete _powerEngine;
  }

  /*!
  */
  void AbstractFKFunction::setPowerEngine(FKPower * p)
  {
    if(p!=_powerEngine) {
      delete _powerEngine;
      _powerEngine=p;
    }
  }

  /*!
    \fn void AbstractFKFunction::setGridStep(double step)

    Mandatory grid step to convert real limits into integer limits
    in setTotalLimits().
  */

  /*!
    Angles are in degrees counted from North.
    \a min can be less that \a max. The range starts at \a min and is counted
    clockwise to \a max, eventually passing over 360 deg.

    Examples:

    \li min=10, max=270: all angles from 10 to 270 are accepted.
    \li min=270, max=10: all angles from 270 to 360 and 0 to 10 are accepted

  */
  void AbstractFKFunction::setAzimuthRange(double min, double max)
  {
    // Forces angles between 0 and 360 deg.
    min=Angle::canonicalDegrees(min);
    max=Angle::canonicalDegrees(max);
    // Angles are swapped because math range is counted counter-clockwise
    _maxAzimuth=Angle::geographicToMath(Angle::degreesToRadians(min));
    _minAzimuth=Angle::geographicToMath(Angle::degreesToRadians(max));
    if(_maxAzimuth<=_minAzimuth) {
      _maxAzimuth+=2.0*M_PI;
    }
    App::log(4, tr("Math angle range for FK map: from %1 to %2\n")
             .arg(_minAzimuth).arg(_maxAzimuth));
  }

  void AbstractFKFunction::setTotalLimits(double squaredOmega)
  {
    _squaredKmaxTotal=squaredOmega*_squaredSlowMax;
    if(_squaredKmaxTotal>_squaredKmaxArray) {
      _squaredKmaxTotal=_squaredKmaxArray;
    }
    _squaredKminTotal=squaredOmega*_squaredSlowMin;
    if(_squaredKminTotal<_squaredKminArray) {
      _squaredKminTotal=_squaredKminArray;
    }
    ASSERT(_gridStep>0.0);
    double invStep=1.0/_gridStep;
    double invStep2=invStep*invStep;
    _gridSquaredKminTotal=_squaredKminTotal*invStep2;
    _gridSquaredKmaxTotal=_squaredKmaxTotal*invStep2;
  }

  /*!
    Return true if \a k is close to the limits (<1% relative).
    \a frequency is used only for reporting.
  */
  bool AbstractFKFunction::isOnLimit(const Vector<double>& k, double frequency) const
  {
    double k2=k.squareLength(0, 1);
    // 1% on squared k corresponds to 0.5% on slowness if the k limit is controled by a maximum slowness
    if(fabs(k2-_squaredKminTotal)<0.01*_squaredKminTotal ||
       fabs(k2-_squaredKmaxTotal)<0.01*_squaredKmaxTotal) {
      APP_LOG(4, tr("  %1 (rad/m)^2 (%5 m/s) at %2 Hz on limits %3 or %4\n")
                          .arg(k2).arg(frequency)
                          .arg(_squaredKminTotal).arg(_squaredKmaxTotal)
                          .arg(2.0*M_PI*frequency/sqrt(k2)))
      return true;
    } else {
      if(_minAzimuth+2.0*M_PI!=_maxAzimuth) { // If all angles are accepted, there is no limit
        double a=k.azimuth(0, 1);
        return fabs(Angle::differenceRadians(a, _minAzimuth))<M_PI/180.0 ||
               fabs(Angle::differenceRadians(a, _maxAzimuth))<M_PI/180.0;
      } else {
        return false;
      }
    }
  }

  /*!
    Ellipticity must be the plain value not the angle.
    Returns true if more peaks have to be added, for instance, several ellipticity
    values for the same slowness.
  */
   void AbstractFKFunction::setAssociatedResults(const Vector<double>&, double power,
                                                 FKPeaks::Value& val) const
   {
     val.ellipticity=std::numeric_limits<double>::quiet_NaN();
     val.verticalNoise=std::numeric_limits<double>::quiet_NaN();
     val.horizontalNoise=std::numeric_limits<double>::quiet_NaN();
     val.power=power;
   }

  /*Point AbstractFKFunction::gradient(const Point& k) const
  {
    if(isInsideLimits(k)) {
      FKPower p(_steering);
      setGradient(k, p);
      return p.gradient();
    } else {
      return 0.0;
    }
  }

  double AbstractFKFunction::concavity(const Point& k) const
  {
    if(isInsideLimits(k)) {
      FKPower p(_steering);
      p.initOneComponent();  // does not hurt for three components
      setGradient(k, p);
      setHessian(k, p);
      return p.concavity();
    } else {
      return 0.0;
    }
  }

  Point AbstractFKFunction::stepDirection(const Point& k) const
  {
    if(isInsideLimits(k)) {
      FKPower p(_steering);
      p.initOneComponent();  // does not hurt for three components
      setGradient(k, p);
      setHessian(k, p);
      if(p.concavity()<0.0) {
        return p.newtonStep();
      }
    }
    return Point();
  }*/

  AbstractFKFunction * AbstractFKFunction::create(ArrayStations::Mode mode,
                                                  const FKParameters * param,
                                                  FKCache * cache)
  {
    TRACE;
    // Check that the components are available for the processing
    switch(mode) {
    case ArrayStations::Vertical:
      switch(param->processType()) {
      case FKParameters::Conventional:
      case FKParameters::Capon:
      case FKParameters::Omni:
      case FKParameters::ActiveCapon:
      case FKParameters::ActiveConventional:
        break;
      default:
        App::log(tr("'%1' processing is not allowed for vertical components.\n")
                 .arg(FKParameters::convertProcessType(param->processType())));
        return nullptr;
      }
      break;
    case ArrayStations::Horizontal:
      switch(param->processType()) {
      case FKParameters::ConventionalRadial:
      case FKParameters::ConventionalTransverse:
      case FKParameters::CaponRadial:
      case FKParameters::CaponTransverse:
      case FKParameters::LDS2:
        break;
      default:
        App::log(tr("'%1' processing is not allowed for horizontal components.\n")
                 .arg(FKParameters::convertProcessType(param->processType())));
        return nullptr;
      }
      break;
    case ArrayStations::ThreeComponents:
      // Everything can be done with three components
      break;
    }
    AbstractFKFunction * f=nullptr;
    switch(param->processType()) {
    case FKParameters::Undefined:
      App::log(tr("Processing type is not defined.\n"));
      return nullptr;
    // Vertical components
    case FKParameters::Conventional:
      f=new ConvFKVertical(cache);
      break;
    case FKParameters::Capon:
      f=new HRFKVertical(cache);
      break;
    case FKParameters::Omni:
      //f=new OmniFKVertical(cache);
      break;
    case FKParameters::ActiveCapon:
      f=new ActiveHRFKVertical(cache);
      break;
    case FKParameters::ActiveConventional:
      f=new ActiveConvFKVertical(cache);
      break;
    // Horizontal components
    case FKParameters::ConventionalRadial:
      f=new ConvFKRadial(cache);
      break;
    case FKParameters::ConventionalTransverse:
      f=new ConvFKTransverse(cache);
      break;
    case FKParameters::CaponRadial:
      f=new HRFKRadial(cache);
      break;
    case FKParameters::CaponTransverse:
      f=new HRFKTransverse(cache);
      break;
    case FKParameters::LDS2:
      f=new HRFKDirectTransverse(cache);
      break;
    // Three components
    case FKParameters::ConventionalRayleigh:
      f=new ConvFKRayleighAll(cache);
      break;
    case FKParameters::RTBF:
      f=new HRFKRayleighSquared(cache);
      break;
    case FKParameters::ARTBF:
      f=new HRFKRayleighAll(cache);
      break;
    case FKParameters::RTBFFixedEll:
      if(param->fixedEllipticityFileName().isEmpty()) {
        App::log(tr("Ellipticity file is not available in parameters.\n"));
      } else {
        HRFKRayleighFixedEll * ff=new HRFKRayleighFixedEll(cache);
        if(ff->loadFixedEllipticityCurve(param->fixedEllipticityFileName())) {
          f=ff;
        } else {
          App::log(tr("Error loading ellipticity file '%1'.\n")
                   .arg(param->fixedEllipticityFileName()));
          delete ff;
        }
      }
      break;
    case FKParameters::ARDS:
      f=new HRFKDirectRayleighAll(cache);
      break;
    case FKParameters::LDS3:
      f=new HRFKDirectLove(cache);
      break;
    case FKParameters::SensorOrientation:
      f=new SensorOrientationCapon(cache);
      break;
    case FKParameters::RDSSingle:
      f=new SingleDirectRayleigh(cache);
      break;
    //case FKParameters::Omni:  // TODO: Omni should be split vertical/three component
    //  f=new OmniFKRayleigh(cache);
    //  break;
    case FKParameters::PoggiVertical:
      f=new PoggiFKVertical(cache);
      break;
    case FKParameters::PoggiRadial:
      f=new PoggiFKRadial(cache);
      break;
    case FKParameters::ActiveConventionalRayleigh:
      f=new ActiveConvFKRayleighAll(cache);
      break;
    case FKParameters::ActiveRTBF:
      f=new ActiveHRFKRayleigh(cache);
      break;
    }
    return f;
  }

  FunctionSearch * AbstractFKFunction::createSearch(const FKParameters& param)
  {
    Q_UNUSED(param);
    return nullptr;
  }

  const ComplexMatrix * AbstractFKFunction::crossSpectrum() const
  {
    if(_crossSpectrum) {
      return _crossSpectrum->matrix();
    } else {
      return nullptr;
    }
  }


} // namespace ArrayCore
