/***************************************************************************
**
**  This file is part of spac2disp.
**
**  spac2disp is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  spac2disp is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2020-11-12
**  Copyright: 2020
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <ArrayCore.h>

#include "Reader.h"
#include "Samples.h"

/*!
  \class Reader Reader.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
Reader::Reader()
  : HistogramReader()
{
  TRACE;
  _samples=nullptr;
  _outputFileName="a.max";
  _sMin=1.0/4000.0;
  _sMax=1.0/50.0;
  _strict=false;
}

/*!
  Description of destructor still missing
*/
Reader::~Reader()
{
  TRACE;
  delete _samples;
}

bool Reader::setOptions(int& argc, char ** argv)
{
  TRACE;
  // Check arguments
  int i, j=1;
  for(i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-o") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _outputFileName=argv[i];
      } else if(arg=="-strict") {
        _strict=true;
      } else {
        argv[j++]=argv[i];
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j<argc) {
    argv[j]=nullptr;
    argc=j;
  }
  return HistogramReader::setOptions(argc, argv);
}

void Reader::help(ApplicationHelp * h)
{
  TRACE;
  h->addGroup("Max file", "max");
  h->addOption("-o <FILE>", "Output file (default='a.max').");
  h->addOption("-strict", "The .target file corresponding to the .max must be also provided. "
                          "For each ring, the minimum of the average is considered as a maximum frequency. "
                          "It avoids contamination of the results with solutions computed for high frequency lobes.");
  h->addOption("-vmin <MIN>", "Sets minimum velocoty (default=50 m/s).");
  h->addOption("-vmax <MAX>", "Sets maximum velocity (default=4000 m/s).");
}

/*!
  Get parameters
*/
bool Reader::parse(QTextStream& s)
{
  TRACE;
  qint64 pos=s.pos();
  QStringList beginPatterns, endPatterns;
  beginPatterns << "# BEGIN PARAMETERS";
  endPatterns << "# END PARAMETERS";
  if(!_parameters.load(s, beginPatterns, endPatterns)) {
    return false;
  }
  s.seek(pos);
  return HistogramReader::parse(s);
}

/*!

*/
bool Reader::loadSamples(QTextStream& s, const QString& lastLine)
{
  TRACE;
  ConsoleProgress progress;
  progress.setCaption(tr("Loading"));
  bool ok=true;
  Sample sample;
  QString line;
  if(File::isDataLine(lastLine)) {
    if(sample.read(lastLine)) {
      _samples->append(sample);
    } else {
      ok=false;
    }
  }
  while(!s.atEnd()) {
    if(File::readLine(s, line)) {
      if(sample.read(line)) {
        _samples->append(sample);
      } else {
        ok=false;
      }
    }
    progress.setValue(_samples->count());
  }
  progress.end(_samples->count());
  return ok;
}

bool Reader::setFrequencyRange(const AutocorrCurves& curves)
{
  TRACE;
  if(_parameters.rings()!=*curves.rings()) {
    App::log(tr("Rings defined in .target file do not match with those found in .max file (parameter section).\n"));
    return false;
  }
  if(_parameters.rings().count()!=curves.curves().count()) {
    App::log(tr("There must be only one curve per ring.\n"));
    return false;
  }
  int n=curves.rings()->count();
  _maxFrequencies.resize(n);
  for(int i=0; i<n; i++) {
    _maxFrequencies[i]=curves.firstBesselMinimum(i);
    App::log(tr("Ring %1 (%2 to %3 m): max frequency=%4\n")
             .arg(i)
             .arg(_parameters.rings().at(i).minRadius())
             .arg(_parameters.rings().at(i).maxRadius())
             .arg(_maxFrequencies[i]));
  }
  return true;
}

int Reader::convert()
{
  TRACE;
  if(_samples->isEmpty()) {
    return 2;
  }
  CoreApplication::instance()->debugUserInterrupts(false);
  ConsoleProgress progress;
  progress.setCaption(tr("Converting to dispersion"));
  _samples->sort();
  AutocorrDispersion eng;
  FKPeaks outPeaks;
  outPeaks.setTimeReference(_samples->first().time());
  DateTime time;
  PrivateVector<double> k(2, 0.0);
  int ns=_samples->count();
  progress.setMaximum(ns-1);
  for(int is=0; is<ns; is++) {
    const Sample& samp=_samples->at(is);
    if(fabs(samp.autocorr())>0.025 && samp.autocorr()>-0.4) {
      WaveNumberConverter conv(samp.frequency());
      conv.setTime(samp.time());
      double omega=2.0*M_PI*conv.frequency();
      int ringIndex=samp.ring();
      if(ringIndex<_maxFrequencies.count()) {
        if(conv.frequency()>=_maxFrequencies.at(ringIndex)) {
          continue;
        }
      }
      eng.setRing(_parameters.rings().at(ringIndex));
      VectorList<double> solutions=eng.dispersion(omega, samp.autocorr(), omega*_sMin, omega*_sMax);
      for(int i=solutions.count()-1; i>=0; i--) {
        k[0]=solutions.at(i)*omega;
        FKPeaks::Value * val=outPeaks.add();
        double slowness=conv.slowness(k);
        if(val && slowness>_sMin) {
          val->time=outPeaks.time(conv.time());
          val->frequency=conv.frequency();
          val->slowness=slowness;
          val->azimuth=std::numeric_limits<double>::quiet_NaN();
          val->ellipticity=std::numeric_limits<double>::quiet_NaN();
          val->verticalNoise=std::numeric_limits<double>::quiet_NaN();
          val->horizontalNoise=std::numeric_limits<double>::quiet_NaN();
          val->power=std::numeric_limits<double>::quiet_NaN();
          val->valid=true;
          val->extraValues=nullptr;
        }
      }
    }
    progress.setValue(is);
  }
  CoreApplication::instance()->debugUserInterrupts(true);
  progress.end(ns-1);
  if(outPeaks.save(_outputFileName, &_parameters)) {
    App::log(tr("Results saved in '%1'.\n").arg(_outputFileName));
    return 0;
  } else {
    return 2;
  }
}
