/***************************************************************************
**
**  This file is part of phaseit.
**
**  phaseit is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  phaseit is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2020-12-07
**  Copyright: 2020
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "Worker.h"
#include "ProcessSignals.h"

/*!
  \class Worker Worker.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
Worker::Worker()
  : ParallelTaskWorker()
{
  TRACE;
}

/*!
  Description of destructor still missing
*/
Worker::~Worker()
{
  TRACE;
}

void Worker::setTaskManager(TaskManager * tm)
{
  ParallelTaskWorker::setTaskManager(tm);
  int i=tm->parameters()->referenceIndex();
  if(i<0 || i>=tm->stations()->count()) {
    App::log(tr("Bad reference index (%1), reset to default (first station)\n").arg(i));
    i=0;
  }
  _reference=new ProcessSignals(tm->stations()->at(i));
}

void Worker::process(ParallelTask * t)
{
  const Parameters * param=taskManager()->parameters();
  int iStat=t->taskIndex()<param->referenceIndex() ? t->taskIndex() : t->taskIndex()+1;
  const StationSignals * stat=taskManager()->stations()->at(iStat);
  t->setStatus(stat->name());
  while(!terminated()) {
    if(!t->hasStepToProcess()) {
      t->setFinishing();
      break;
    } else {
      int windowIndex=t->currentStep();
      t->addSteps(1);
      t->beginProcess(); // Avoid removal of task
      t->unlock();

      const TimeRange& tw=taskManager()->timeWindows()->at(windowIndex);
      ProcessSignals * procStat=new ProcessSignals(stat);
      _reference->setProcessed(tw, nullptr);
      procStat->setProcessed(tw, nullptr);
     if(param->domain()==Parameters::Frequency) {
       calculateFrequency(procStat, param, t->taskIndex(), windowIndex);
     } else if(param->domain()==Parameters::Time) {
       calculateTime(procStat, param, t->taskIndex(), windowIndex);
     }
     delete procStat;

      t->lock();
      t->endProcess();
    }
  }
  t->unlock();
}

void Worker::calculateFrequency(ProcessSignals * procStat, const Parameters * param,
                                int resIndex, int windowIndex)
{
  DoubleSignal * refSig=_reference->processed(0);
  DoubleSignal * sig=procStat->processed(0);
  refSig->fastFourierTransform(DoubleSignal::Spectrum);
  sig->fastFourierTransform(DoubleSignal::Spectrum);
  const double * refSamples=refSig->constLockSamples();
  const double * samples=sig->constLockSamples();

  FrequencyBand freq;
  freq.setRelativeWidth(param->frequencyBandwidth());
  int nBands=param->frequencySampling().count();
  for(int iBand=0; iBand<nBands; iBand++) {
    if(terminated()) break;
    freq.setCenter(param->frequencySampling().value(iBand));
    GaussianFrequencyBand frequencyFilter;
    frequencyFilter.calculate(freq, sig->duration(), sig->samplingFrequency());
    int nFreq=frequencyFilter.maximumIndex();
    Complex crossSpectrum;
    double refPower=0.0, power=0.0;
    for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
      Complex cref=refSig->complex(refSamples, iFreq);
      Complex c=sig->complex(samples, iFreq);
      double filter=frequencyFilter.taperValue(iFreq);
      refPower+=cref.abs2()*filter;
      power+=c.abs2()*filter;
      c.conjugate();
      c*=cref;
      c*=filter;
      crossSpectrum+=c;
    }
    double valCoherence=100.0*crossSpectrum.abs2()/(refPower*power);
    double valPhase=100.0*crossSpectrum.phase()/(2.0*M_PI);
    double valAmp=power/refPower;
    taskManager()->addResult(resIndex, windowIndex, iBand, valCoherence, valPhase, valAmp);
  }
  refSig->unlockSamples();
  sig->unlockSamples();
}

void Worker::calculateTime(ProcessSignals * procStat, const Parameters * param,
                           int resIndex, int windowIndex)
{
  DoubleSignal * refSig=_reference->processed(0);
  DoubleSignal * sig=procStat->processed(0);
  FilterParameters filterParam;
  filterParam.setMethod(FilterParameters::FrequencyWindow);
  filterParam.frequencyWindow().setShape(WindowFunctionParameters::Gaussian);
  filterParam.setBand(FilterParameters::BandPass);
  int nFreqs=param->frequencySampling().count();
  for(int iBand=0; iBand<nFreqs; iBand++) {
    if(terminated()) break;
    double freq=param->frequencySampling().value(iBand);
    // Filter
    filterParam.frequencyWindow().setSigma(0.34161); // Same shape as in GaussianFrequencyBand
    filterParam.setFrequencyRange(freq*(1.0-param->frequencyBandwidth()),
                                  freq*(1.0+param->frequencyBandwidth()));
    DoubleSignal * refSigFilt=new DoubleSignal(*refSig);
    refSigFilt->copySamplesFrom(refSig);
    refSigFilt->fastFourierTransform(DoubleSignal::Spectrum);
    refSigFilt->filter(filterParam);
    double refPower=refSigFilt->power(filterParam.minimumFrequency(), filterParam.maximumFrequency());
    refSigFilt->fastFourierTransform(DoubleSignal::Waveform);
    DoubleSignal * sigFilt=new DoubleSignal(*sig);
    sigFilt->copySamplesFrom(sig);
    sigFilt->fastFourierTransform(DoubleSignal::Spectrum);
    sigFilt->filter(filterParam);
    double sigPower=sigFilt->power(filterParam.minimumFrequency(), filterParam.maximumFrequency());
    sigFilt->fastFourierTransform(DoubleSignal::Waveform);
    DoubleSignal * corr=new DoubleSignal;
    corr->normalizedCorrelation(refSigFilt, sigFilt, param->maximumDelay());
    DoubleSignal * fineCorr=new DoubleSignal(*corr);
    //double fineOverSample=1.0e5/fineCorr->samplingFrequency();
    int fineOverSample=1000;
    fineCorr->overSample(fineOverSample, corr);
    //corr->debugSave(QString("corr_%1_%2_%3").arg(stat->name()).arg(tw.center()).arg(freq));
    //fineCorr->debugSave(QString("fineCorr_%1_%2_%3").arg(stat->name()).arg(tw.center()).arg(freq));
    int dt=fineCorr->maximumAt(0, fineCorr->nSamples()-1);

    double valCoherence=100.0*fineCorr->value(dt);
    double valPhase=100.0*freq*fineCorr->samplingPeriod()*(dt-0.5*(fineCorr->nSamples()-fineOverSample));
    double valAmp=sigPower/refPower;
    taskManager()->addResult(resIndex, windowIndex, iBand, valCoherence, valPhase, valAmp);

    delete refSigFilt;
    delete sigFilt;
    delete corr;
    delete fineCorr;
  }
}
