/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-16
**  Copyright: 2018-2019

**
***************************************************************************/

#include <SciFigs.h>

#include "gpviewmaxVersion.h"
#include "gpviewmaxInstallPath.h"
#include "HistogramsWidget.h"
#include "Reader.h"

PACKAGE_INFO("gpviewmax", GPVIEWMAX)

ApplicationHelp * help();

HistogramsWidget * createWidgets(Reader& reader)
{
  HistogramsWidget * w=new HistogramsWidget;
  w->setNormalize(reader.isNormalized());
  w->addWidget(reader.samples(), reader);
  w->setHistogramComments(reader.fileNames().join(";"));
  if(reader.fileNames().count()==1) {
    QFileInfo fi(reader.fileNames().first());
    w->setWindowTitle(fi.fileName()+" - gpviewmax");
  }
  if(!w->splitModes(reader)) {
    delete w;
    return nullptr;
  }
  return w;
}

int main(int argc, char ** argv)
{
  Application a(argc, argv, help);

  // Options
  ExportOptions exportOptions;
  Reader reader;
  if(!reader.setOptions(argc, argv)) {
    return 2;
  }
  if(!exportOptions.read(argc, argv)) {
    return 2;
  }
  if(!CoreApplication::checkRemainingArgs(argc, argv)) {
    return 2;
  }
  reader.setSamples(new Samples);
  if(!reader.read(argc, argv, reader.ignoreStdin())) {
    return 2;
  }
  if(reader.isEmpty() && !a.isBatch()) {
    if(!reader.read(Message::getOpenFileNames(tr("Open data file"),
                                              tr("All files (*);;"
                                                 "Max files (*.max)")))) {
      return 2;
    }
  }
  // User basic filtering
  Samples * samples;
  if(!reader.undefinedEllipticities().value()) {
    samples=reader.samples()->filterUndefinedEllipticities();
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.curveFileNames().isEmpty()) {
    samples=reader.samples()->filterNoiseSmart(reader.noiseDeviation().value(),
                                               reader.slownessDeviation().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
    samples=reader.samples()->filterPowerSmart(reader.powerDeviation().value(),
                                               reader.slownessDeviation().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  samples=reader.samples()->filterRelativePower(reader.relativePowerThreshold());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }
  samples=reader.samples()->filterMinimumVelocity(reader.minimumVelocity().value());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }
  samples=reader.samples()->filterMaximumVelocity(reader.maximumVelocity().value());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }
  samples=reader.samples()->filterMinimumWavenumber(reader.minimumWavenumber().value());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }
  samples=reader.samples()->filterMaximumWavenumber(reader.maximumWavenumber().value());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }
  if(reader.minimumEllipticity().isSet()) {
    samples=reader.samples()->filterMinimumEllipticities(reader.minimumEllipticity().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.maximumEllipticity().isSet()) {
    samples=reader.samples()->filterMaximumEllipticities(reader.maximumEllipticity().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.minimumVerticalNoise().isSet()) {
    samples=reader.samples()->filterMinimumVerticalNoise(reader.minimumVerticalNoise().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.maximumVerticalNoise().isSet()) {
    samples=reader.samples()->filterMaximumVerticalNoise(reader.maximumVerticalNoise().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.minimumHorizontalNoise().isSet()) {
    samples=reader.samples()->filterMinimumHorizontalNoise(reader.minimumHorizontalNoise().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.maximumHorizontalNoise().isSet()) {
    samples=reader.samples()->filterMaximumHorizontalNoise(reader.maximumHorizontalNoise().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.minimumTotalNoise().isSet()) {
    samples=reader.samples()->filterMinimumTotalNoise(reader.minimumTotalNoise().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.maximumTotalNoise().isSet()) {
    samples=reader.samples()->filterMaximumTotalNoise(reader.maximumTotalNoise().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.minimumDeltaNoise().isSet()) {
    samples=reader.samples()->filterMinimumDeltaNoise(reader.minimumDeltaNoise().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.maximumDeltaNoise().isSet()) {
    samples=reader.samples()->filterMaximumDeltaNoise(reader.maximumDeltaNoise().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.minimumSigmaNoise().isSet()) {
    samples=reader.samples()->filterMinimumSigmaNoise(reader.minimumSigmaNoise().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.maximumSigmaNoise().isSet()) {
    samples=reader.samples()->filterMaximumSigmaNoise(reader.maximumSigmaNoise().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.minimumAzimuth().isSet()) {
    samples=reader.samples()->filterMinimumAzimuth(reader.minimumAzimuth().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.maximumAzimuth().isSet()) {
    samples=reader.samples()->filterMaximumAzimuth(reader.maximumAzimuth().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.minimumPower().isSet()) {
    samples=reader.samples()->filterMinimumPower(reader.minimumPower().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.maximumPower().isSet()) {
    samples=reader.samples()->filterMaximumPower(reader.maximumPower().value());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }

  if(!reader.setDefaultLimits()) {
    return 2;
  }

  int appReturn;
  switch(reader.action()) {
  case HistogramReader::Gui: {
      SciFigsGlobal s(true);
      HistogramsWidget * w=createWidgets(reader);
      if(!w) {
        return 2;
      }
      if(!w->classifySamples(reader.filters())) {
        return 2;
      }
      w->exportPlot(exportOptions);
      w->show();
      appReturn=a.exec();
      delete w;
    }
    break;
  case HistogramReader::MaxFile: {
      reader.samples()->save(reader.exportMaxFileName());
      appReturn=0;
    }
    break;
  case HistogramReader::MinimumVelocity: {
      Histogram2D * h=reader.histogram(reader.dispersionSampling());
      h->setYScaleType(Scale::InverseLog);
      h->setSamples(reader.samples()->dispersionSamples());

      Curve<RealStatisticalPoint> c=h->pickAll(4, 0.1, 0.2);
      QList<Curve<RealStatisticalPoint>> l=c.split(1.2, LogScale, 0.1, 5);
      int c0=0, i0=0;
      double s0=0.0;
      for(int j=l.count()-1; j>=0; j--) {
        const Curve<RealStatisticalPoint>& c=l.at(j);
        int n=c.count();
        for(int i=0; i<n; i++) {
          const RealStatisticalPoint& p=c.constAt(i);
          double s=::exp(::log(p.mean())+2.0*::log(p.stddev()));
          if(s>s0) {
            c0=j;
            i0=i;
            s0=s;
          }
        }
      }
      QTextStream(stdout) << tr("Minimum velocity at %1 Hz (curve %2 point %3) = %4 m/s (mean-2*stddev on a log scale)\n")
                             .arg(l.at(c0).x(i0)).arg(c0).arg(i0).arg(1.0/s0);
      appReturn=0;
    }
    break;
  case HistogramReader::Plot: {
      SciFigsGlobal s(true);
      HistogramsWidget * w=createWidgets(reader);
      if(!w) {
        return 2;
      }
      if(!w->classifySamples(reader.filters())) {
        return 2;
      }
      w->exportPlot(exportOptions);
      appReturn=0;
      delete w;
    }
    break;
  case HistogramReader::Misfit: {
      bool ok=true;
      Curve<Point2D> m=reader.misfit(ok);
      if(ok) {
        QTextStream(stdout) << "# misfit\n"
                            << m.toString();
        appReturn=0;
      } else {
        appReturn=2;
      }
    }
    break;
  case HistogramReader::HistogramValues: {
      SciFigsGlobal s(true);
      HistogramsWidget * w=createWidgets(reader);
      if(!w) {
        return 2;
      }
      if(!w->classifySamples(reader.filters())) {
        return 2;
      }
      w->exportHistogramValues(exportOptions);
      appReturn=0;
      delete w;
    }
    break;
  default:
    appReturn=2;
    break;
  }
  return appReturn;
}

ApplicationHelp * help()
{
  TRACE;
  ApplicationHelp * h=new ApplicationHelp;
  h->setOptionSummary( "[OPTIONS] <FILE>...");
  h->setComments("Plot all values of a .max file generated with three-component FK. "
                 "Plot of individual modes.");
  h->addGroup("Main", "main");
  h->addOption("-dots", "Plots data samples as transparent dots. They can be categorized for modal identification.");
  h->addOption("-histograms", "Plots histograms of data samples (default).");
  h->addOption("-rayleigh", "Plots Rayleigh ellipticity and noise, null ellipticities are filtered "
                     "out unless option '-undef-ell' is specified (default=automatically detected).");
  h->addOption("-split-ell-sign", "Split postive and negative ellipticities.");
  h->addOption("-auto-pick", "Automatically pick curves.");
  h->addOption("-undef-ell", "Keep samples with undefined ellipticities (default=filter them).");
  h->addGroup("filter", "filter");
  h->addOption("-rel-power <R>", "Accept only samples with power > R*maximum power (over a time window and for a frequency).");
  h->addOption("-curve-rel-range <R>", "Keep R*mean around curves to split modes (default=0.2).");
  h->addOption("-low-noise <N> <S>", "Keep only low noise samples in a slowness neighborhood. "
                                     "For a given frequency, each sample S0, N0 is considered individually. "
                                     "If there is at least one sample Si, Ni in the slowness neighborhood "
                                     "defined by [S0/(1+S), S0*(1+S)] for which Ni is lower than N0/(1+N), "
                                     "the current sample is filtered out.");
  h->addOption("-high-power <P> <S>", "Keep only high power samples in a slowness neighborhood. "
                                     "For a given frequency, each sample S0, P0 is considered individually. "
                                     "If there is at least one sample Si, Pi in the slowness neighborhood "
                                     "defined by [S0/(1+S), S0*(1+S)] for which Pi is higher than P0*(1+P), "
                                     "the current sample is filtered out.");
  h->addOption("-filter <PARAM>", "Filter histograms according to PARAM file. Several instances are processed "
                                  "in the command line order.");
  h->addOption("-filter-example", "Prints a filter parameter example to stdout.");
  h->addOption("-v-min <V>", "Filter out all samples have a velocity below V.");
  h->addOption("-v-max <V>", "Filter out all samples have a velocity above V.");
  h->addOption("-k-min <K>", "Filter out all samples have a wavenumber below K.");
  h->addOption("-k-max <K>", "Filter out all samples have a wavenumber above K.");
  h->addGroup("Output", "output");
  h->addOption("-plot", "Export plot (see '-h plot' for options).");
  h->addOption("-max <FILE>", "Output filter max values and exit.");
  h->addOption("-min-velocity", "Output minimum velocity decreased by two standard deviations and exit.");
  h->addOption("-misfit <TYPE>", "Returns the misfit curve of the sample population compared to loaded "
                                 "curves (see -curve). TYPE must be either 'Slowness' or "
                                 "'Ellipticity'.");
  h->addOption("-hist-values", "Export the histogram values for the selected plot (or the first one if none is selected). "
                               "See '-h plot' to learn how to select plots and to define exported file name.");
  HistogramReader::helpCurves(h);
  HistogramReader::helpX(h);
  Reader::helpY(h);
  h->addGroup("Plot", "plot", 2);
  ExportOptions::help(h);
  return h;
}
