/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-16
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef READER_H
#define READER_H

#include <QGpCoreStat.h>

#include "Samples.h"

class Reader : public HistogramReader
{
public:
  Reader();
  ~Reader();

  enum PlotType {Histograms, Dots};
  enum Plot {Dispersion=1, Ellipticity=2,
             VerticalNoise=4, HorizontalNoise=8, TotalNoise=16,
             DeltaNoise=32, SigmaNoise=64,
             Azimuth=128, Power=256};
  Q_DECLARE_FLAGS(Plots, Plot)

  bool setOptions(int& argc, char ** argv);
  bool setDefaultLimits();

  bool isEmpty() const {return _samples->isEmpty();}

  static void helpY(ApplicationHelp * h);

  void setSamples(Samples * s) {_samples=s;}
  Samples * samples() const {return _samples;}

  PlotType plotType() const {return _plotType;}

  const Option<double>& relativeRange() const {return _relativeRange;}
  const Option<bool>& undefinedEllipticities() const {return _undefinedEll;}
  const Option<bool>& splitEllipticitySign() const {return _splitEllSign;}
  const Option<bool>& autoPick() const {return _autoPick;}
  const Option<double>& noiseDeviation() const {return _noiseDev;}
  const Option<double>& powerDeviation() const {return _powerDev;}
  const Option<double>& slownessDeviation() const {return _slownessDev;}

  const Option<double>& minimumVelocity() const {return _minV;}
  const Option<double>& maximumVelocity() const {return _maxV;}
  const Option<double>& minimumWavenumber() const {return _minK;}
  const Option<double>& maximumWavenumber() const {return _maxK;}
  const Option<double>& minimumEllipticity() const {return _minEll;}
  const Option<double>& maximumEllipticity() const {return _maxEll;}
  const Option<double>& minimumVerticalNoise() const {return _minVerticalNoise;}
  const Option<double>& maximumVerticalNoise() const {return _maxVerticalNoise;}
  const Option<double>& minimumHorizontalNoise() const {return _minHorizontalNoise;}
  const Option<double>& maximumHorizontalNoise() const {return _maxHorizontalNoise;}
  const Option<double>& minimumTotalNoise() const {return _minTotalNoise;}
  const Option<double>& maximumTotalNoise() const {return _maxTotalNoise;}
  const Option<double>& minimumDeltaNoise() const {return _minDeltaNoise;}
  const Option<double>& maximumDeltaNoise() const {return _maxDeltaNoise;}
  const Option<double>& minimumSigmaNoise() const {return _minSigmaNoise;}
  const Option<double>& maximumSigmaNoise() const {return _maxSigmaNoise;}
  const Option<double>& minimumAzimuth() const {return _minAzimuth;}
  const Option<double>& maximumAzimuth() const {return _maxAzimuth;}
  const Option<double>& minimumPower() const {return _minPower;}
  const Option<double>& maximumPower() const {return _maxPower;}

  const SamplingParameters& dispersionSampling() const {return _dispersionSampling;}
  const SamplingParameters& ellipticitySampling() const {return _ellipticitySampling;}
  const SamplingParameters& verticalNoiseSampling() const {return _verticalNoiseSampling;}
  const SamplingParameters& horizontalNoiseSampling() const {return _horizontalNoiseSampling;}
  const SamplingParameters& totalNoiseSampling() const {return _totalNoiseSampling;}
  const SamplingParameters& deltaNoiseSampling() const {return _deltaNoiseSampling;}
  const SamplingParameters& sigmaNoiseSampling() const {return _sigmaNoiseSampling;}
  const SamplingParameters& azimuthSampling() const {return _azimuthSampling;}
  const SamplingParameters& powerSampling() const {return _powerSampling;}

  const QString& dispersionTitle() const {return _dispersionTitle;}
  const QString& ellipticityTitle() const {return _ellipticityTitle;}
  const QString& verticalNoiseTitle() const {return _verticalNoiseTitle;}
  const QString& horizontalNoiseTitle() const {return _horizontalNoiseTitle;}
  const QString& totalNoiseTitle() const {return _totalNoiseTitle;}
  const QString& deltaNoiseTitle() const {return _deltaNoiseTitle;}
  const QString& sigmaNoiseTitle() const {return _sigmaNoiseTitle;}
  const QString& azimuthTitle() const {return _azimuthTitle;}
  const QString& powerTitle() const {return _powerTitle;}

  const QStringList& filters() const {return _filters;}

  const QString& referenceDispersionLayer() const {return _referenceDispersionLayer;}
  const QString& referenceEllipticityLayer() const {return _referenceEllipticityLayer;}

  double arrayKmin() const {return _arrayKmin;}
  double arrayKmax() const {return _arrayKmax;}

  Plots showPlots() const {return _showPlots;}

  Histogram2D * histogram(const SamplingParameters& ySampling) const;
  Curve<Point2D> misfit(bool& ok) const {return HistogramReader::misfit(ok);}
protected:
  bool parse(QTextStream& s);
  bool loadSamples(QTextStream& s, const QString& lastLine);
  void misfit(const QList<Curve<Point2D>>& curves, Curve<Point2D>& m,
              int * hitCount, double precision) const;
  void setRayleigh();

  PlotType _plotType;
  Plots _showPlots;
  union {
    struct {
      ManualFlags dispersion;
      ManualFlags ellipticity;
      ManualFlags verticalNoise;
      ManualFlags horizontalNoise;
      ManualFlags totalNoise;
      ManualFlags deltaNoise;
      ManualFlags sigmaNoise;
      ManualFlags azimuth;
      ManualFlags power;
    };
    int flags[5];
  } _manual;
private:
  Samples * _samples;

  Option<double> _relativeRange;
  Option<bool> _undefinedEll, _splitEllSign, _autoPick;
  Option<double> _noiseDev, _powerDev, _slownessDev;
  Option<double> _minK, _maxK;
  Option<double> _minV, _maxV;
  Option<double> _minEll, _maxEll;
  Option<double> _minVerticalNoise, _maxVerticalNoise;
  Option<double> _minHorizontalNoise, _maxHorizontalNoise;
  Option<double> _minTotalNoise, _maxTotalNoise;
  Option<double> _minDeltaNoise, _maxDeltaNoise;
  Option<double> _minSigmaNoise, _maxSigmaNoise;
  Option<double> _minAzimuth, _maxAzimuth;
  Option<double> _minPower, _maxPower;
  double _arrayKmin, _arrayKmax;

  SamplingParameters _dispersionSampling;
  SamplingParameters _ellipticitySampling;
  SamplingParameters _verticalNoiseSampling;
  SamplingParameters _horizontalNoiseSampling;
  SamplingParameters _totalNoiseSampling;
  SamplingParameters _deltaNoiseSampling;
  SamplingParameters _sigmaNoiseSampling;
  SamplingParameters _azimuthSampling;
  SamplingParameters _powerSampling;
  QString _dispersionTitle;
  QString _ellipticityTitle;
  QString _verticalNoiseTitle;
  QString _horizontalNoiseTitle;
  QString _totalNoiseTitle;
  QString _deltaNoiseTitle;
  QString _sigmaNoiseTitle;
  QString _azimuthTitle;
  QString _powerTitle;
  QString _referenceDispersionLayer;
  QString _referenceEllipticityLayer;
  QStringList _filters;

  enum MisfitType {SlownessMisfit, EllipticityMisfit};
  MisfitType _misfitType;
};

#endif // READER_H

