/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-16
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HistogramsWidget.h"
#include "ModeWidgets.h"

/*!
  \class HistogramsWidget HistogramsWidget.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
HistogramsWidget::HistogramsWidget(QWidget * parent)
  : QWidget(parent)
{
  TRACE;
  setupUi(this);
  setAttribute(Qt::WA_DeleteOnClose, false);

  Settings::getWidget(this, "Histogram");

  _normalize=true;

  connect(sheet->sheet(), SIGNAL(activeSelectionChanged(GraphicObject *)),
          this, SLOT(setActivePlot(GraphicObject *)));
  QAction * a=curves->addCurveAction(tr("&Adjust"), tr("Adjust current curve to the closest maximum"), true);
  connect(a, SIGNAL(triggered()), this, SLOT(adjustCurve()));

  // Init the curve menu
  QMenu * m=new QMenu(this);
  m->setTitle(tr("Curve"));
  m->setTearOffEnabled (true);
  a=new QAction(tr("&Mean"), this);
  m->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(addMeanCurve()));
  a=new QAction(tr("Medi&an"), this);
  m->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(addMedianCurve()));
  a=new QAction(tr("Mo&de"), this);
  m->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(addModeCurve()));
  a=new QAction(tr("&Gaussian mixture"), this);
  m->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(addGaussianMixtureCurve()));
  curveBut->setMenu(m);

  // Init the sample menu
  m=new QMenu(this);
  m->setTitle(tr("Samples"));
  m->setTearOffEnabled (true);
  a=new QAction(tr("&Classify"), this);
  m->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(classifySamples()));
  a=new QAction(tr("&Export"), this);
  m->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(exportSamples()));
  a=new QAction(tr("&Pick to mean"), this);
  m->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(pickToMean()));
  samplesBut->setMenu(m);
}

/*!
  Description of destructor still missing
*/
HistogramsWidget::~HistogramsWidget()
{
  TRACE;
}

void HistogramsWidget::closeEvent(QCloseEvent * e)
{
  TRACE;
  Settings::setWidget(this, "Histogram");
  e->accept();
}

void HistogramsWidget::addWidget(Samples * s, const Reader& param)
{
  TRACE;
  ModeWidgets * w=new ModeWidgets(this);
  w->setPlotType(param.plotType());
  _modes.append(w);

  w->addWidgets(sheet->sheet(), param);
  w->setSamples(s, param);
  w->setLimits(_normalize);
  w->initCurves(curves);
}

void HistogramsWidget::setHistogramComments(QString cmt)
{
  TRACE;
  for(QList<ModeWidgets *>::iterator it=_modes.begin(); it!=_modes.end(); it++) {
    (*it)->setComments(cmt);
  }
}

bool HistogramsWidget::splitModes(const Reader& param)
{
  TRACE;
  if(param.curveFileNames().isEmpty()) {
    if(param.splitEllipticitySign().value()) {
      splitModesFromEllSign(param);
    }
  } else {
    return splitModesFromCurves(param);
  }
  return true;
}

bool HistogramsWidget::splitModesFromCurves(const Reader& param)
{
  DispersionProxy proxy;
  ColumnTextParser parser;
  parser.setStandardTypes(proxy.columnFileTypes());
  if(param.curveFormatFileName().isEmpty()) {
    parser.setTypes(proxy.defaultColumnFileTypes());
  } else {
    XMLHeader hdr(&parser);
    if(hdr.xml_restoreFile(param.curveFormatFileName())!=XMLClass::NoError) {
      return false;
    }
  }
  QStringList expandedFileNames=File::expand(param.curveFileNames());
  for(QStringList::const_iterator itf=expandedFileNames.begin(); itf!=expandedFileNames.end(); itf++) {
    QFile f(*itf);
    QFileInfo fi(*itf);
    if(f.open(QIODevice::ReadOnly)) {
      parser.setText(new QTextStream(&f));
      parser.start();
      parser.wait();
      ColumnTextIterator it(&parser);
      int curveNumber=0;
      while(!it.atEnd()) {
        ModalCurve c;
        proxy.setCurve(&c);
        if(!proxy.parse(it)) {
          return false;
        }
        c.setName(fi.fileName()+"#"+QString::number(++curveNumber));
        Samples * newSamples=param.samples()->filterCurve(c, param.relativeRange().value());
        Samples * filtSamples=newSamples->filterNoiseSmart(param.noiseDeviation().value(),
                                                           param.slownessDeviation().value());
        if(filtSamples) {
          delete newSamples;
          newSamples=filtSamples;
        }
        addWidget(newSamples, param);
      }
    } else {
      App::log(tr("Cannot open file '%1'\n").arg(*itf));
      return false;
    }
  }
  return true;
}

void HistogramsWidget::splitModesFromEllSign(const Reader& param)
{
  Samples * newSamples, * filtSamples;
  // Negative ellipticities
  newSamples=param.samples()->filterMaximumEllipticities(0.0);
  filtSamples=newSamples->filterNoiseSmart(param.noiseDeviation().value(),
                                           param.slownessDeviation().value());
  if(filtSamples) {
    delete newSamples;
    newSamples=filtSamples;
  }
  addWidget(newSamples, param);
  // Positive ellipticities
  newSamples=param.samples()->filterMinimumEllipticities(0.0);
  filtSamples=newSamples->filterNoiseSmart(param.noiseDeviation().value(),
                                           param.slownessDeviation().value());
  if(filtSamples) {
    delete newSamples;
    newSamples=filtSamples;
  }
  addWidget(newSamples, param);
}

void HistogramsWidget::exportPlot(const ExportOptions& o)
{
  TRACE;
  if(!o.colorMapFile().isEmpty()) {
    ColorMap pal;
    XMLErrorReport xmler(XMLErrorReport::Read | XMLErrorReport::NoMessageBox);
    xmler.setTitle(tr("Load palette"));
    xmler.setFileName(o.colorMapFile());
    XMLSciFigs s;
    if(xmler.exec(s.restoreFile(o.colorMapFile(), &pal, XMLSciFigs::Data))) {
      for(QList<ModeWidgets *>::iterator it=_modes.begin(); it!=_modes.end(); it++) {
        (*it)->setColorMap(pal);
      }
    }
  }
  o.selectObjects(sheet->sheet());
  o.addLayers(sheet->sheet());
  o.restoreMakeUp(sheet->sheet());
  if(!o.preserveScales()) {
    for(QList<ModeWidgets *>::iterator it=_modes.begin(); it!=_modes.end(); it++) {
      (*it)->setLimits(_normalize);
    }
  }
  o.exportObjects(sheet->sheet());
}

void HistogramsWidget::exportHistogramValues(const ExportOptions& o)
{
  TRACE;
  o.selectObjects(sheet->sheet());
  o.exportGridValues(sheet->sheet());
}

ModeWidgets * HistogramsWidget::mode(AxisWindow * w) const
{
  for(QList<ModeWidgets *>::const_iterator it=_modes.begin(); it!=_modes.end(); it++) {
    if((*it)->contains(w)) {
      return *it;
    }
  }
  return nullptr;
}

ModeWidgets * HistogramsWidget::currentMode() const
{
  if(!curves->currentLayer()) {
    AxisWindow * w=_modes.first()->dispersionPickLayer()->graph();
    sheet->sheet()->select(w);
  }
  return mode(curves->currentLayer()->graph());
}

void HistogramsWidget::setActivePlot(GraphicObject * o)
{
  TRACE;
  AxisWindow * w=qobject_cast<AxisWindow *>(o);
  if(!w) {
    curves->setCurrentLayer(nullptr);
    return;
  }
  ModeWidgets * mw=mode(w);
  if(mw) {
    LineLayer * layer=mw->pickLayer(w);
    if(layer) {
      curves->setProxy(mw->curveProxy(w));
      curves->setPlotProxy(mw->curvePlotProxy(w));
      curves->setCurrentLayer(layer);
    }
  }
}

/*!

*/
void HistogramsWidget::classifySamples()
{
  TRACE;
  ModeWidgets * w=currentMode();
  if(w) {
    w->classifySamples(_normalize);
  }
}

bool HistogramsWidget::classifySamples(const QStringList& filters)
{
  ClassificationParameters param;
  for(QStringList::const_iterator itp=filters.begin(); itp!=filters.end(); itp++) {
    if(param.load(*itp)) {
      APP_LOG(2, param.toString());
      ColumnTextParser * parser=nullptr;
      for(QList<ModeWidgets *>::const_iterator itm=_modes.begin(); itm!=_modes.end(); itm++) {
        switch(param.data()) {
        case Reader::Dispersion:
          setActivePlot((*itm)->dispersionPickLayer()->graph());
          break;
        case Reader::Ellipticity:
          setActivePlot((*itm)->ellipticityPickLayer()->graph());
          break;
        case Reader::VerticalNoise:
          setActivePlot((*itm)->verticalNoisePickLayer()->graph());
          break;
        case Reader::HorizontalNoise:
          setActivePlot((*itm)->horizontalNoisePickLayer()->graph());
          break;
        case Reader::TotalNoise:
          setActivePlot((*itm)->totalNoisePickLayer()->graph());
          break;
        case Reader::DeltaNoise:
          setActivePlot((*itm)->deltaNoisePickLayer()->graph());
          break;
        case Reader::SigmaNoise:
          setActivePlot((*itm)->sigmaNoisePickLayer()->graph());
          break;
        case Reader::Azimuth:
          setActivePlot((*itm)->azimuthPickLayer()->graph());
          break;
        case Reader::Power:
          setActivePlot((*itm)->powerPickLayer()->graph());
          break;
        }
        if(!parser && !param.file().isEmpty()) {
          parser=new ColumnTextParser;
          parser->setStandardTypes(curves->proxy()->columnFileTypes());
          XMLHeader hdr(parser);
          if(hdr.xml_restoreFile(param.fileParser())!=XMLClass::NoError) {
            delete parser;
            return false;
          }
        }
        if(!curves->loadMultiColumns(param.file(), true, parser)) {
          delete parser;
          return false;
        }
        App::log(tr("File '%1' loaded: %2 curves available\n")
                 .arg(param.file())
                 .arg(curves->count()));
        (*itm)->classifySamples(param, _normalize);
        curves->clear();
      }
      delete parser;
    } else {
      return false;
    }
  }
  return true;
}

void HistogramsWidget::adjustCurve()
{
  TRACE;

  int index=curves->currentLine();
  ModeWidgets * w=currentMode();
  if(index>=0 && w) {
    CurvePropertiesWidget * properties=curves->curves().at(index);
    CurveProxy * proxy=properties->proxy();
    if(proxy->sampleCount()==0) return;
    proxy->sort();
    properties->beginCurveChange();
    w->adjust();
    proxy->addLog(tr("Adjust curve to histogram closest modes\n"));
    properties->endCurveChange();
  }
}

void HistogramsWidget::addMeanCurve()
{
  TRACE;
  ModeWidgets * w=currentMode();
  if(w) {
    w->addMeanCurve();
  }
}

void HistogramsWidget::addMedianCurve()
{
  TRACE;
  ModeWidgets * w=currentMode();
  if(w) {
    w->addMedianCurve();
  }
}

void HistogramsWidget::addModeCurve()
{
  TRACE;
  ModeWidgets * w=currentMode();
  if(w) {
    w->addModeCurve();
  }
}

void HistogramsWidget::addGaussianMixtureCurve()
{
  TRACE;
  ModeWidgets * w=currentMode();
  if(w) {
    w->addGaussianMixtureCurve();
  }
}

void HistogramsWidget::exportSamples()
{
  TRACE;
  ModeWidgets * w=currentMode();
  if(w) {
    w->exportSamples();
  }
}

void HistogramsWidget::pickToMean()
{
  TRACE;
  ModeWidgets * w=currentMode();
  if(w) {
    w->pickToMean();
  }
}
