/***************************************************************************
**
**  This file is part of gpviewdcreport.
**
**  gpviewdcreport is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewdcreport is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-12-05
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <DinverCore.h>
#include <DinverDCCore.h>
#include <DinverDCGui.h>
#include <QGpCoreTools.h>
#include <QGpCoreWave.h>
#include <QGpGuiTools.h>
#include "gpviewdcreportVersion.h"
#include "gpviewdcreportInstallPath.h"

PACKAGE_INFO("gpviewdcreport", GPVIEWDCREPORT)

ApplicationHelp * help();

int main(int argc, char ** argv)
{
  Application a(argc, argv, help);
  SciFigsGlobal s(true);

  // Options
  ExportOptions exportOptions;
  enum PlotType {Profile, Dispersion, Ellipticity, Autocorr, RefraVp, RefraVs};
  PlotType plotType=Profile;
  bool skipDialog=false;
  int nGraphs=1;
  double maxMisfit=1.0;
  int minIndex=0;
  int maxIndex=0;
  QString targetFile;
  Mode mode;
  // Check arguments
  int i, j=1;
  for(i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-profile") {
        plotType=Profile;
      } else if(arg=="-dispersion") {
        plotType=Dispersion;
      } else if(arg=="-ellipticity") {
        plotType=Ellipticity;
      } else if(arg=="-autocorr") {
        plotType=Autocorr;
      } else if(arg=="-refraVp") {
        plotType=RefraVp;
      } else if(arg=="-refraVs") {
        plotType=RefraVs;
      } else if(arg=="-m") {
        CoreApplication::checkOptionArg(i, argc, argv);
        maxMisfit=CoreApplication::toDouble(i, i-1, argv);
        skipDialog=true;
      } else if(arg=="-min-index") {
        CoreApplication::checkOptionArg(i, argc, argv);
        minIndex=CoreApplication::toInt(i, i-1, argv);
        skipDialog=true;
      } else if(arg=="-max-index") {
        CoreApplication::checkOptionArg(i, argc, argv);
        maxIndex=CoreApplication::toInt(i, i-1, argv);
        skipDialog=true;
      } else if(arg=="-n") {
        CoreApplication::checkOptionArg(i, argc, argv);
        nGraphs=CoreApplication::toInt(i, i-1, argv);
        skipDialog=true;
      } else if(arg=="-s") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString s(argv[i]);
        bool ok=true;
        mode.setSlowness(Mode::convertSlowness(s, ok));
        if(!ok) {
          App::log(tr("gpviewdcreport: bad slowness type '%1'\n").arg(argv[i]));
          return false;
        }
        skipDialog=true;
      } else if(arg=="-p") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString p(argv[i]);
        bool ok=true;
        mode.setPolarization(Mode::convertPolarization(p, ok));
        if(!ok) {
          App::log(tr("gpviewdcreport: bad polarization '%1'\n").arg(argv[i]));
          return false;
        }
        skipDialog=true;
      } else if(arg=="-abs") {
        mode.setValue(Mode::Absolute);
        skipDialog=true;
      } else if(arg=="-target") {
        Application::checkOptionArg(i, argc, argv);
        targetFile=argv[i];
      } else {
        argv[j++]=argv[i];
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j < argc) {
    argv[j]=nullptr;
    argc=j;
  }
  if(!exportOptions.read(argc, argv)) {
    return 2;
  }
  if(!Application::checkRemainingArgs(argc, argv)) {
    return 2;
  }

  QStringList reportList;
  i=1;
  while(i<argc) {
    reportList.append(argv[i]);
    i++;
  }
  if(reportList.isEmpty()) {
    reportList=Message::getOpenFileNames(tr("View report files"),
                                            tr("Report file (*.report)"));
    if(reportList.isEmpty()) {
      App::log(tr("gpviewdcreport: no report files provided\n") );
      return 2;
    }
  }
  DCModelViewer * w;
  SAFE_UNINITIALIZED(w,nullptr)
  if(skipDialog) {
    switch(plotType) {
    case Profile:
      w=new GroundModelViewer;
      break;
    case Dispersion:
      w=new DispersionViewer;
      w->setMode(mode);
      break;
    case Ellipticity:
      w=new EllipticityViewer;
      w->setMode(mode);

      break;
    case Autocorr:
      w=new AutocorrViewer;
      w->setMode(mode);
      w->setFrequencyAxis(true);
      break;
    case RefraVp:
      w=new RefractionVpViewer;
      break;
    case RefraVs:
      w=new RefractionVsViewer;
      break;
    }
    w->initGraphs(nGraphs);
    for(QStringList::iterator it=reportList.begin();it!=reportList.end();++it) {
      w->selectModels(*it, maxMisfit, minIndex, maxIndex);
      QFileInfo fi(*it);
      w->setWindowTitle(w->windowTitle()+" "+fi.completeBaseName());
    }
    w->loadModels();
    if(!exportOptions.colorMapFile().isEmpty()) {
      w->setColorMap(exportOptions.firstColorMap());
    }
  } else {
    ViewerParam * d=new ViewerParam();
    switch(plotType) {
    case Profile:
      w=d->groundProfile(reportList);
      break;
    case Dispersion:
      w=d->dispersion(reportList);
      break;
    case Ellipticity:
      w=d->ellipticity(reportList);
      break;
    case Autocorr:
      w=d->autocorr(reportList);
      break;
    case RefraVp:
      w=d->refractionVp(reportList);
      break;
    case RefraVs:
      w=d->refractionVs(reportList);
      break;
    }
    delete d;
  }
  if(!w) return 2;
  if(!targetFile.isEmpty()) {
    TargetList * tl=new TargetList;
    XMLVirtualPlugin plugin(tl, "DispersionCurve");
    XMLDinverHeader hdr(&plugin);
    if(hdr.xml_restoreFile(targetFile)!=XMLClass::NoError) {
      delete w;
      delete tl;
      return 2;
    }
    w->setTarget(tl);
  }

  w->exportPlot(exportOptions);
  if(!exportOptions.exportFile().isEmpty()) {
    return 0;
  }
  if(skipDialog) {
    w->show();
  }

  int appReturn=a.exec();
  return appReturn;
}

ApplicationHelp * help()
{
  ApplicationHelp * h=new ApplicationHelp;
  h->setOptionSummary( "[OPTIONS] [FILE]..." );
  h->setComments( "Display ground profiles, dispersion curves, ellipticity curves, autocorrelation curves or refraction curves "
                  "read from .report files. It is strictly equivalent to the 'view' options in dinver graphical interface. "
                  "If options '-m', '-n' or other related options are specified the initial dialog box is skipped." );
  h->addGroup("gpviewdcreport", "gpviewdcreport");
  h->addOption("-profile", "Plot ground profiles (default)");
  h->addOption("-dispersion", "Plot dispersion curves");
  h->addOption("-ellipticity", "Plot ellipticity curves");
  h->addOption("-autocorr", "Plot autocorrelation curves");
  h->addOption("-refraVp", "Plot refraction curves for Vp");
  h->addOption("-refraVs", "Plot refraction curves for Vs");
  h->addOption("-m <MISFIT>","Maximum misfit to display");
  h->addOption("-min-index <INDEX>","Minimum index to display (default=first)");
  h->addOption("-max-index <INDEX>","Maximum misfit to display (default=last)");
  h->addOption("-target <TARGET>","Set target shown above models read from .report files.");
  h->addGroup("Profile","profile");
  h->addOption("-n <N>", "Number of profiles");
  h->addGroup("Dispersion","dispersion");
  h->addOption("-n <N>", "Number of modes");
  h->addOption("-s <SLOWNESS>", "Slowness type: Group or Phase");
  h->addOption("-p <POLARISATION>", "Polarization: Rayleigh or Love");
  h->addGroup("Ellipticity","ellipticity");
  h->addOption("-n <N>", "Number of modes");
  h->addOption("-abs", "Plot absolute ellipticity.");
  h->addGroup("Autocorr","autocorr");
  h->addOption("-n <N>", "Number of rigns");
  h->addOption("-p <POLARISATION>", "Polarization: Vertical, Transverse or Radial");
  h->addGroup("Refra","refra");
  h->addOption("-n <N>", "Number of sources");
  h->addGroup("Plot", "plot", 2);
  ExportOptions::help(h);
  return h;
}
