/***************************************************************************
**
**  This file is part of gphistogram.
**
**  gphistogram is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gphistogram is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-12-18
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <SciFigs.h>
#include <QGpGuiStat.h>

#include "HistogramWidget.h"

/*!
  \class HistogramWidget HistogramWidget.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
HistogramWidget::HistogramWidget(QWidget * parent)
    : QWidget(parent),
      _gaussianMixtureInversion(1)
{
  TRACE;
  setupUi(this);
  setAttribute(Qt::WA_DeleteOnClose, false);
  setFocusPolicy(Qt::ClickFocus);
  _hist=nullptr;
  _normalize=true;
  _lastX=0;
  _gaussianMixtureCurve=nullptr;
  _gaussianMixtureOutput.setHistogramInversion(&_gaussianMixtureInversion);
  connect(&_gaussianMixtureInversion, SIGNAL(histogramFinished(int)),
          this, SLOT(addGaussianMixture(int)));

  AxisWindow * w;
  QAction * a;

  a=curves->addCurveAction(tr("&Adjust"), tr("Adjust current curve to the closest maximum"), true);
  connect(a, SIGNAL(triggered()), this, SLOT(adjustCurve()));

  w=new AxisWindow(this);
  w->setObjectName("Histogram");
  sheet->sheet()->addObject(w);
  w->setPrintLeft(0.5);
  w->setPrintTop(0.5);
  w->setPrintWidth(16.0);
  w->setPrintHeight(8.0);
  w->updateInternalGeometry();
  _currentXLayer=new ParallelBands(w);
  _currentXLayer->addBand(0.0, Pen(Qt::black));
  _histogramLayer=new IrregularGrid2DPlot(w);
  _histogramLayer->setSmooth(false); // Currently a shift is observed for smoothed data
                                     // TODO: check the smooth display algorithm
  ColorMap map;
  map.generate(0, 19, ColorPalette::defaultSequentialWhite, ColorPalette::Reversed);
  map.setWhiteTransparent(true);
  _histogramLayer->setColorMap(map);
  LineLayer * curveLayer=new LineLayer(w);
  curveLayer->setReferenceLine(new RealStatisticalLine);
  curves->setProxy(new RealStatisticalProxy);
  curves->setPlotProxy(new RealStatisticalPlotProxy);
  curves->initLayer(curveLayer);
  curves->setCurrentLayer(curveLayer);
  //a=curves->addCurveAction(tr("&Adjust"), tr("Adjust current curve to the closest maximum"), true);
  //connect(a, SIGNAL(triggered()), this, SLOT(adjustCurve()) );
  sheet->sheet()->showObject(w);

  new QShortcut(Qt::Key_Up, this, SLOT(nextX()));
  new QShortcut(Qt::Key_Up | Qt::SHIFT, this, SLOT(nextXReject()));
  new QShortcut(Qt::Key_Down, this, SLOT(previousX()));
  new QShortcut(Qt::Key_Down | Qt::SHIFT, this, SLOT(previousXReject()));
  new QShortcut(Qt::Key_Left, this, SLOT(decreaseLowLimit()));
  new QShortcut(Qt::Key_Left | Qt::SHIFT, this, SLOT(decreaseHighLimit()));
  new QShortcut(Qt::Key_Right, this, SLOT(increaseLowLimit()));
  new QShortcut(Qt::Key_Right | Qt::SHIFT, this, SLOT(increaseHighLimit()));

  // Init the curve menu
  QMenu * m=new QMenu(this);
  m->setTitle(tr( "Curve"));
  m->setTearOffEnabled (true);
  curveBut->setMenu(m);

  a=new QAction(tr("&Mean"), this);
  curveBut->menu()->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(addMeanCurve()));
  a=new QAction(tr("Medi&an"), this);
  curveBut->menu()->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(addMedianCurve()));
  a=new QAction(tr("Mo&de"), this);
  curveBut->menu()->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(addModeCurve()));
  a=new QAction(tr("&Gaussian mixture point"), this);
  curveBut->menu()->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(addGaussianMixturePoint()));
  a=new QAction(tr("Gaussian mixture &curve"), this);
  curveBut->menu()->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(addGaussianMixtureCurve()));
  a=new QAction(tr("&Pick"), this);
  curveBut->menu()->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(pick()));
  a=new QAction(tr("&Pick and split"), this);
  curveBut->menu()->addAction(a);
  connect(a, SIGNAL(triggered()), this, SLOT(pickAndSplit()));

  // Density plot
  w=new AxisWindow(this);
  sheet->sheet()->addObject(w);
  w->setPrintLeft(0.5);
  w->setPrintTop(8.5);
  w->setPrintWidth(8.0);
  w->setPrintHeight(6.0);
  w->updateInternalGeometry();
  _densityBandLayer=new ParallelBands(w);
  _densityBandLayer->setObjectName("DensitySelection");
  _densityBandLayer->addTrackingAction(tr("Pick bands"), 0, tr("Define the X band to reject."));
  _densityBandLayer->setOpacity(0.6);
  _densityBandLayer->addBand(0, 0, Brush(QColor( 205, 242, 255)));
  connect(_densityBandLayer, SIGNAL(bandPicked(double, double)), this, SLOT(setBand( double, double)));
  _densityHistogramLayer=new HistogramLayer(w);
  _densityHistogramLayer->setObjectName("DensityHistogram");
  _densityHistogramLayer->setReferenceLine(new PlotLine2D);
  _densityHistogramLayer->addLine(Pen(Qt::blue), Symbol(Symbol::NoSymbol, 1.0, Pen(Qt::NoPen), QColor(152, 152, 255)));
  _densityCurveLayer=new LineLayer(w);
  _densityCurveLayer->setObjectName("DensityCurve");
  _densityCurveLayer->setReferenceLine(new PlotLine2D);
  _densityCurveLayer->setReferencePen(Pen(Qt::blue));
  _densityCurveLayer->addLine(Pen(Qt::red), Symbol());
  _densityCurveLayer->addLine(Pen(Qt::green), Symbol());
  w->yAxis()->setNumberType('e');
  w->yAxis()->setNumberPrecision(1);
  w->yAxis()->setTitle("Probability density");
  w->yAxis()->setZoomEnabled(false);
  sheet->sheet()->showObject(w);

  w=new AxisWindow(this);
  sheet->sheet()->addObject(w);
  w->setPrintLeft(8.5);
  w->setPrintTop(8.5);
  w->setPrintWidth(8.0);
  w->setPrintHeight(6.0);
  w->updateInternalGeometry();
  _cumulativeBandLayer=new ParallelBands(w);
  _cumulativeBandLayer->setObjectName("CumulativeSelection");
  _cumulativeBandLayer->addTrackingAction(tr("Pick bands"), 0, tr("Define the X band to reject."));
  _cumulativeBandLayer->setOpacity(0.6);
  _cumulativeBandLayer->addBand(0, 0, Brush(QColor( 205, 242, 255)));
  connect(_cumulativeBandLayer, SIGNAL(bandPicked(double, double)), this, SLOT(setBand( double, double)));
  _cumulativeCurveLayer=new LineLayer(w);
  _cumulativeCurveLayer->setObjectName("Cumulative");
  _cumulativeCurveLayer->setReferenceLine(new PlotLine2D);
  _cumulativeCurveLayer->setReferencePen(Pen( Qt::blue));
  _cumulativeCurveLayer->addLine(Pen(Qt::blue), Symbol());
  _cumulativeCurveLayer->addLine(Pen(Qt::red), Symbol());
  _cumulativeCurveLayer->addLine(Pen(Qt::green), Symbol());
  w->yAxis()->setRange(0.0, 1.0);
  w->yAxis()->setNumberType('f');
  w->yAxis()->setNumberPrecision(1);
  w->yAxis()->setTitle("Cumulative probability");
  w->yAxis()->setZoomEnabled(false);
  sheet->sheet()->showObject(w);

  Settings::getWidget(this, "Histogram");
}

/*!
  Description of destructor still missing
*/
HistogramWidget::~HistogramWidget()
{
  TRACE;
}

void HistogramWidget::closeEvent(QCloseEvent * e)
{
  TRACE;
  Settings::setWidget(this, "Histogram");
  e->accept();
}

void HistogramWidget::setHistogramComments(QString cmt)
{
  TRACE;
  _histogramLayer->setComments(cmt);
}

void HistogramWidget::setHistogram(Histogram2D * hist)
{
  TRACE;
  _hist=hist;
  if(_normalize) {
    _hist->normalizeArea(YAxis);
  }
  _histogramLayer->setGrid(*_hist);

  if(!_hist->xTitle().isNull()) {
    _histogramLayer->graph()->xAxis()->setTitle(_hist->xTitle());
    _histogramLayer->graph()->xAxis()->setTitleInverseScale(_hist->xTitleInverseScale());
  }
  if(!_hist->yTitle().isNull()) {
    _histogramLayer->graph()->yAxis()->setTitle(_hist->yTitle());
    _histogramLayer->graph()->yAxis()->setTitleInverseScale(_hist->yTitleInverseScale());
    _densityHistogramLayer->graph()->xAxis()->setTitle(_hist->yTitle());
    _densityHistogramLayer->graph()->xAxis()->setTitleInverseScale(_hist->yTitleInverseScale());
    _cumulativeCurveLayer->graph()->xAxis()->setTitle(_hist->yTitle());
    _cumulativeCurveLayer->graph()->xAxis()->setTitleInverseScale(_hist->yTitleInverseScale());
  }

  _histogramLayer->graph()->xAxis()->setScaleType(hist->xScaleType());
  switch(hist->yScaleType()) {
  case Scale::Linear:
  case Scale::Inverse:
  case Scale::AbsoluteTime:
  case Scale::RelativeTime:
    break;
  case Scale::Log:
  case Scale::InverseLog:
    curves->currentLayer()->setErrorBar(LineLayer::VerticalLogBar);
    break;
  }
  _histogramLayer->graph()->yAxis()->setScaleType(hist->yScaleType());
  _densityHistogramLayer->graph()->xAxis()->setScaleType(hist->yScaleType());
  _cumulativeCurveLayer->graph()->xAxis()->setScaleType(hist->yScaleType());

  if(hist->isPeriodic()) {
    // Implies that periodic values are centered around 0
    curves->currentLayer()->setSignThreshold(hist->period()*0.25);
  }
  _gaussianMixtureInversion.setHistogram(0, _hist);

  xScroll->setRange(0, hist->nx()-1);
  on_xScroll_valueChanged();
  _histogramLayer->graph()->deepUpdate();
}

void HistogramWidget::setLimits()
{
  TRACE;
  GraphContents * gc=_histogramLayer->graphContents();
  AxisWindow * w=gc->graph();

  Rect r=_histogramLayer->boundingRect();
  r.enlarge(0.01, gc->scaleX().sampling(), gc->scaleY().sampling());
  w->xAxis()->setRange(r.x1(), r.x2());
  w->yAxis()->setRange(r.y1(), r.y2());

  ColorMap pal=_histogramLayer->colorMap();
  double m=_histogramLayer->grid().maximumValue();
  if(_normalize) {
    pal.setValues(0.0001*m, m, SamplingParameters::Log);
  } else {
    pal.generate(0, qRound(m), ColorPalette::defaultSequential, ColorPalette::Reversed);
    pal.setValues(0.5, m-0.5, SamplingParameters::Linear);
  }
  _histogramLayer->setColorMap(pal);
}

void HistogramWidget::on_xScroll_valueChanged()
{
  TRACE;
  int ix=xScroll->value();
  double x=_hist->x(ix);
  _currentXLayer->band(0).setValue(x);
  _currentXLayer->deepUpdate();
  if(_xUnit.isEmpty()) {
    xEdit->setText(QString::number(x));
  } else {
    xEdit->setText(QString("%1 %2").arg(x).arg(_xUnit));
  }

  PlotLine2D * gridDensity=static_cast<PlotLine2D *>(_densityHistogramLayer->line(0));
  PlotLine2D * gaussDensityLine=static_cast<PlotLine2D *>(_densityCurveLayer->line(0));
  PlotLine2D * gaussMixDensityLine=static_cast<PlotLine2D *>(_densityCurveLayer->line(1));
  PlotLine2D * gridCumulative=static_cast<PlotLine2D *>(_cumulativeCurveLayer->line(0));
  PlotLine2D * gaussCumulativeLine=static_cast<PlotLine2D *>(_cumulativeCurveLayer->line(1));
  PlotLine2D * gaussMixCumulativeLine=static_cast<PlotLine2D *>(_cumulativeCurveLayer->line(2));

  _densityCurveLayer->lockDelayPainting();
  _densityHistogramLayer->lockDelayPainting();
  _cumulativeCurveLayer->lockDelayPainting();

  Histogram densityCurve=_hist->histogramAt(ix);
  gridDensity->setCurve(densityCurve);
  gridCumulative->setCurve(densityCurve.cumulative());

  // Compute average directly on samples (exact)
  RealStatisticalValue meanStat=_hist->mean(ix);
  RealStatisticalValue medStat=densityCurve.median();
  RealStatisticalValue modStat=densityCurve.mode();

  double mean, stddev;
  if(_hist->ySampling() & LogScale) {
    mean=::log(meanStat.mean());
    stddev=::log(meanStat.stddev());
  } else if(_hist->ySampling() & InverseScale) {
    mean=1.0/meanStat.mean();
    stddev=1.0/meanStat.stddev();
  } else {
    mean=meanStat.mean();
    stddev=meanStat.stddev();
  }
  double period=0.0, halfPeriod=0.0;
  if(_hist->isPeriodic()) {
    period=_hist->y(_hist->ny()-1)-_hist->y(0);
    halfPeriod=0.5*period;
  }

  int nSimulated=10*densityCurve.count();
  int firstIndex=0, lastIndex=densityCurve.count()-1;
  densityCurve.firstValid(firstIndex);
  densityCurve.lastValid(lastIndex);
  densityCurve.resample(nSimulated, densityCurve.constAt(firstIndex).x(),
                        densityCurve.constAt(lastIndex).x(), _hist->ySampling());
  gaussDensityLine->setCurve(densityCurve);
  gaussCumulativeLine->setCurve(densityCurve);
  gaussMixDensityLine->setCurve(densityCurve);
  gaussMixCumulativeLine->setCurve(densityCurve);

  NormalDistribution gd(mean, stddev);
  GaussianMixtureDistribution mixd=_gaussianMixtureInversion.distribution(ix);
  for(int i=nSimulated-1; i>=0; i--) {
    Point2D& gaussDens=gaussDensityLine->curve().constXAt(i);
    Point2D& gaussCum=gaussCumulativeLine->curve().constXAt(i);
    Point2D& mixDens=gaussMixDensityLine->curve().constXAt(i);
    Point2D& mixCum=gaussMixCumulativeLine->curve().constXAt(i);
    double x=gaussDens.x();
    if(_hist->ySampling() & LogScale) {
      gaussDens.setY(gd.value(::log(x)));
      gaussCum.setY(gd.cumulativeValue(log(x)));
      mixDens.setY(mixd.value1D(0, ::log(x)));
      mixCum.setY(mixd.cumulativeValue1D(0, ::log(x)));
    } else if(_hist->ySampling() & InverseScale) {
      gaussDens.setY(gd.value(1.0/x));
      gaussCum.setY(gd.cumulativeValue(1.0/x));
      mixDens.setY(mixd.value1D(0, 1.0/x));
      mixCum.setY(mixd.cumulativeValue1D(0, 1.0/x));
    } else {
      if(_hist->isPeriodic()) {
        PeriodicStatistics::shiftFunction(x, mean, period, halfPeriod);
      }
      gaussDens.setY(gd.value(x));
      gaussCum.setY(gd.cumulativeValue(x));
      mixDens.setY(mixd.value1D(0, x));
      mixCum.setY(mixd.cumulativeValue1D(0, x));
    }
  }

  const ParallelBand& b=_densityBandLayer->band(0);
  saveBand(_lastX, b.minimum(), b.maximum());
  Point2D p=restoreBand(ix);
  if(std::isnormal(p.x()) && std::isnormal(p.y())) {
    setBand(p.x(), p.y());
  } else {
    if(_hist->ySampling() & LogScale) {
      setBand(log(mean-stddev), log(mean+stddev));
    } else if(_hist->ySampling() & InverseScale) {
      setBand(1.0/(mean-stddev), 1.0/(mean+stddev));
    } else {
      setBand(mean-stddev, mean+stddev);
    }
  }
  _lastX=ix;

  _densityCurveLayer->unlock();
  _densityHistogramLayer->unlock();
  _cumulativeCurveLayer->unlock();

  statValuesLabel->setText(tr("Mean %2%1 - Stddev %3%1\n"
                              "Median %4%1 - Meddev %5%1\n"
                              "Mode %6%1 - Moddev %7%1\n"
                              "Cumulative median misfit %10")
                            .arg(_yUnit.isEmpty() ? _yUnit : " " + _yUnit)
                            .arg(meanStat.mean(), 0, 'g', 6)
                            .arg(meanStat.stddev(), 0, 'g', 6)
                            .arg(medStat.mean(), 0, 'g', 6)
                            .arg(medStat.stddev(), 0, 'g', 6)
                            .arg(modStat.mean(), 0, 'g', 6)
                            .arg(modStat.stddev(), 0, 'g', 6)
                            .arg(_hist->misfit(_hist->x(ix), gd)));

  Rect r;
  GraphContents * gc;
  AxisWindow * w;

  gc=_densityBandLayer->graphContents();
  w=gc->graph();
  r=gc->boundingRect();
  w->xAxis()->setRange(r.x1(), r.x2());
  w->yAxis()->setRange(r.y1(), r.y2());
  w->deepUpdate();

  gc=_cumulativeBandLayer->graphContents();
  w=gc->graph();
  r=gc->boundingRect();
  w->xAxis()->setRange(r.x1(), r.x2());
  w->deepUpdate();
}

void HistogramWidget::saveBand(int ix, double min, double max)
{
  TRACE;
  double x=_hist->x(ix);
  CurvePropertiesWidget * lowProp=curves->find("Low limit");
  if(lowProp) {
    lowProp->beginCurveChange();
  } else if(std::isnormal(min)) {
    lowProp=curves->addLine(curves->currentLayer()->addLine());
    lowProp->beginCurveChange();
    lowProp->proxy()->setName("Low limit");
  }
  CurvePropertiesWidget * highProp=curves->find("High limit");
  if(highProp) {
    highProp->beginCurveChange();
  } else if(std::isnormal(max)) {
    highProp=curves->addLine(curves->currentLayer()->addLine());
    highProp->beginCurveChange();
    highProp->proxy()->setName("High limit");
  }

  if(lowProp && highProp) {
    Curve<RealStatisticalPoint>& lowCurve=static_cast<RealStatisticalProxy *>(lowProp->proxy())->curve();
    lowCurve.insert(RealStatisticalPoint(x, min));
    Curve<RealStatisticalPoint>& highCurve=static_cast<RealStatisticalProxy *>(highProp->proxy())->curve();
    highCurve.insert(RealStatisticalPoint(x, max));
    lowProp->endCurveChange();
    highProp->endCurveChange();
  }
}

Point2D HistogramWidget::restoreBand(int ix)
{
  TRACE;
  double x=_hist->x(ix);
  CurvePropertiesWidget * lowProp=curves->find("Low limit");
  CurvePropertiesWidget * highProp=curves->find("High limit");
  if(!lowProp || ! highProp) {
    return Point2D(NAN, NAN);
  }
  Curve<RealStatisticalPoint>& lowCurve=static_cast<RealStatisticalProxy *>(lowProp->proxy())->curve();
  Curve<RealStatisticalPoint>& highCurve=static_cast<RealStatisticalProxy *>(highProp->proxy())->curve();
  return Point2D(lowCurve.interpole(x, _hist->xSampling(), _hist->ySampling()).y(),
                 highCurve.interpole(x, _hist->xSampling(), _hist->ySampling()).y());
}

void HistogramWidget::setBand(double min, double max)
{
  TRACE;
  ParallelBand& b1=_densityBandLayer->band(0);
  b1.setMinimum(min);
  b1.setMaximum(max);
  ParallelBand& b2=_cumulativeBandLayer->band(0);
  b2.setMinimum(min);
  b2.setMaximum(max);
  _densityBandLayer->graphContents()->deepUpdate();
  _cumulativeBandLayer->graphContents()->deepUpdate();
}

void HistogramWidget::on_saveBut_clicked()
{
  TRACE;
  QString fileName=Message::getSaveFileName(tr("Save filtered samples"),
                                            tr("Max file (*.max)"));
  if(!fileName.isEmpty()) {
    _hist->save(fileName, _headerLine, true);
  }
}

/*!
  Reject all samples outside the band. If SHIFT key is pressed, the current x is incremented.
  If the band for next x does not exist, a new one is created from the current one plus a translation
  calculated on the median difference before the rejection at the current frequency.
*/
void HistogramWidget::on_rejectBut_clicked()
{
  TRACE;
  int ix=xScroll->value();
  const ParallelBand& band=_densityBandLayer->band(0);
  Rect r(_hist->left(ix), -std::numeric_limits<double>::infinity(), _hist->right(ix), band.minimum());
  _hist->setValid(r, false);
  r.setY1(band.maximum());
  r.setY2(std::numeric_limits<double>::infinity());
  _hist->setValid(r, false);
  _hist->countSamples();
  saveBand(ix, band.minimum(), band.maximum());
  setHistogram(_hist);
}

void HistogramWidget::on_undoBut_clicked()
{
  TRACE;
  int ix=xScroll->value();
  Rect r(_hist->left(ix), -std::numeric_limits<double>::infinity(), _hist->right(ix), std::numeric_limits<double>::infinity());
  _hist->setValid(r, true);
  _hist->countSamples();
  setHistogram(_hist);
}

void HistogramWidget::on_rejectAllBut_clicked()
{
  TRACE;
  Rect r;
  for(int i=_hist->nx()-1; i>=0; i-- ) {
    Point2D band=restoreBand(i);
    r.setLimits(_hist->left(i), -std::numeric_limits<double>::infinity(), _hist->right(i), band.x());
    _hist->setValid(r, false);
    r.setY1(band.y());
    r.setY2(std::numeric_limits<double>::infinity());
    _hist->setValid(r, false);
  }
  _hist->countSamples();
  setHistogram(_hist);
}

void HistogramWidget::on_undoAllBut_clicked()
{
  TRACE;
  Rect r( -std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::infinity(), std::numeric_limits<double>::infinity());
  _hist->setValid(r, true);
  _hist->countSamples();
  setHistogram(_hist);
}

void HistogramWidget::shortcutEvents(Shortcuts s)
{
  TRACE;
  GraphContents * gc=_densityHistogramLayer->graphContents();
  const ParallelBand& b=_densityBandLayer->band(0);
  int ix=xScroll->value();

  double dx;
  SAFE_UNINITIALIZED(dx,0);
  switch (s) {
  case NextX:
  case NextXReject:
  case PreviousX:
  case PreviousXReject:
    break;
  case DecreaseLowLimit:
  case IncreaseLowLimit:
    dx=fabs(b.minimum()-gc->options().xs2r(gc->options().xr2s(b.minimum())+5)); // estimation of 5 pixel width
    break;
  case DecreaseHighLimit:
  case IncreaseHighLimit:
    dx=fabs(b.maximum()-gc->options().xs2r(gc->options().xr2s(b.maximum())+5)); // estimation of 5 pixel width
    break;
  }

  switch (s) {
  case NextX:
    xScroll->setValue(ix+1);
    break;
  case NextXReject:
    on_rejectBut_clicked();
    xScroll->setValue(ix+1);
    break;
  case PreviousX:
    xScroll->setValue(ix-1);
    break;
  case PreviousXReject:
    on_rejectBut_clicked();
    xScroll->setValue(ix-1);
    break;
  case DecreaseLowLimit:
    setBand(b.minimum()-dx, b.maximum());
    break;
  case DecreaseHighLimit:
    setBand(b.minimum(), b.maximum()-dx);
    break;
  case IncreaseLowLimit:
    setBand(b.minimum()+dx, b.maximum());
    break;
  case IncreaseHighLimit:
    setBand(b.minimum(), b.maximum()+dx);
    break;
  }
}

void HistogramWidget::adjustCurve()
{
  TRACE;
  int index=curves->currentLine();
  CurvePropertiesWidget * properties=curves->curves().at(index);
  CurveProxy * proxy=properties->proxy();
  properties->beginCurveChange();
  proxy->sort();
  proxy->followMaximumX(_hist, proxy->minimumX(), proxy->maximumX());
  proxy->addLog(tr("Adjust curve to histogram closest modes\n"));
  properties->endCurveChange();
}

void HistogramWidget::addCurve(const Curve<RealStatisticalPoint>& c)
{
  TRACE;
  RealStatisticalLine * l=static_cast<RealStatisticalLine *>(curves->currentLayer()->addLine());
  l->setCurve(c);
  curves->addLine(l);
}

void HistogramWidget::addMeanCurve()
{
  TRACE;
  RealStatisticalCurve c;
  RealStatisticalProxy p;
  p.setCurve(&c);
  _hist->meanCurve(&p);
  addCurve(c);
}

void HistogramWidget::addMedianCurve()
{
  TRACE;
  RealStatisticalCurve c;
  RealStatisticalProxy p;
  p.setCurve(&c);
  _hist->medianCurve(&p);
  addCurve(c);
}

void HistogramWidget::addModeCurve()
{
  TRACE;
  RealStatisticalCurve c;
  RealStatisticalProxy p;
  p.setCurve(&c);
  _hist->modeCurve(&p);
  addCurve(c);
}

void HistogramWidget::setGaussianMixtureParameters(const GaussianMixtureParameters& p)
{
  TRACE;
  _gaussianMixtureInversion.setParameters(p);
}

void HistogramWidget::addGaussianMixturePoint()
{
  TRACE;
  addCurve(Curve<RealStatisticalPoint>());
  _gaussianMixtureCurve=curves->curves().last();
  _gaussianMixtureCurve->setEditable(false);
  _gaussianMixtureCurve->line()->setPen(Pen(Qt::NoPen));
  _gaussianMixtureCurve->line()->setSymbol(Symbol(Symbol::Circle, 1.2, Pen::SolidLine, Qt::SolidPattern));
  _gaussianMixtureInversion.start(xScroll->value());
}

void HistogramWidget::addGaussianMixtureCurve()
{
  TRACE;
  addCurve(Curve<RealStatisticalPoint>());
  _gaussianMixtureCurve=curves->curves().last();
  _gaussianMixtureCurve->setEditable(false);
  _gaussianMixtureCurve->line()->setPen(Pen(Qt::NoPen));
  _gaussianMixtureCurve->line()->setSymbol(Symbol(Symbol::Circle, 1.2, Pen::SolidLine, Qt::SolidPattern));
  _gaussianMixtureInversion.start();
}

void HistogramWidget::addGaussianMixture(int index)
{
  TRACE;
  if(!_gaussianMixtureCurve) {
    return;
  }
  GaussianMixtureDistribution d=_gaussianMixtureOutput.filter(index);
  double x=_hist->x(index);
  SamplingOptions ySampling=_hist->ySampling();
  RealStatisticalPoint p;
  _gaussianMixtureCurve->beginCurveChange();
  Curve<RealStatisticalPoint>& c=static_cast<RealStatisticalLine *>(_gaussianMixtureCurve->line())->curve();
  for(int i=0; i<d.modeCount(); i++) {
    p.setX(x);
    if(ySampling & LogScale) {
      p.setMean(exp(d.mode(i).mean(0)));
      p.setStddev(exp(d.mode(i).stddev(0)));
    } else if(ySampling & InverseScale) {
      p.setMean(1.0/d.mode(i).mean(0));
      p.setStddev(1.0/d.mode(i).stddev(0));
    } else {
      p.setMean(d.mode(i).mean(0));
      p.setStddev(d.mode(i).stddev(0));
    }
    p.setWeight(d.weight(i));
    c.append(p);
  }
  _gaussianMixtureCurve->endCurveChange();
  _histogramLayer->graph()->deepUpdate();
  on_xScroll_valueChanged();
}

void HistogramWidget::pick()
{
  TRACE;
  addCurve(Curve<RealStatisticalPoint>());
  CurvePropertiesWidget * prop=curves->curves().last();
  RealStatisticalLine * line=static_cast<RealStatisticalLine *>(prop->line());
  line->setPen(Pen(Qt::NoPen));
  line->setSymbol(Symbol(Symbol::Circle, 1.2, Pen::SolidLine, Qt::SolidPattern));
  prop->beginCurveChange();
  line->setCurve(_hist->pickAll(4, 0.1, 0.2));
  prop->endCurveChange();
}

void HistogramWidget::pickAndSplit()
{
  TRACE;
  Curve<RealStatisticalPoint> pc=_hist->pickAll(4, 0.1, 0.2);
  QList<Curve<RealStatisticalPoint>> curves=pc.split(1.2, LogScale, 0.1, 5);
  int n=curves.count();
  for(int i=0; i<n; i++) {
    addCurve(curves.at(i));
  }
}

bool HistogramWidget::loadCurves(const QStringList& fileNames, const QString& formatFileName)
{
  TRACE;
  ColumnTextParser parser;
  parser.setStandardTypes(curves->proxy()->columnFileTypes());
  if(formatFileName.isEmpty()) {
    parser.setTypes(curves->proxy()->defaultColumnFileTypes());
  } else {
    XMLHeader hdr(&parser);
    if(hdr.xml_restoreFile(formatFileName)!=XMLClass::NoError) {
      return false;
    }
  }
  ColumnTextParser * pparser=&parser;
  QStringList expandedFileNames=File::expand(fileNames);
  for(QStringList::const_iterator it=expandedFileNames.begin(); it!=expandedFileNames.end(); it++) {
    if(!curves->loadMultiColumns(*it, false, pparser)) {
      return false;
    }
  }
  return true;
}

void HistogramWidget::exportPlot(const ExportOptions& o)
{
  TRACE;
  AxisWindow * w=_histogramLayer->graph();
  if(!o.legendFile().isEmpty()) {
    Legend l;
    XMLErrorReport xmler(XMLErrorReport::Read | XMLErrorReport::NoMessageBox);
    xmler.setTitle(tr("Load legend"));
    xmler.setFileName(o.legendFile());
    XMLSciFigs s;
    if(xmler.exec(s.restoreFile(o.legendFile(), &l, XMLSciFigs::Data))) {
      curves->setLegend(l);
    }
  }
  if(!o.colorMapFile().isEmpty()) {
    ColorMap pal;
    XMLErrorReport xmler(XMLErrorReport::Read | XMLErrorReport::NoMessageBox);
    xmler.setTitle(tr("Load palette"));
    xmler.setFileName(o.colorMapFile());
    XMLSciFigs s;
    if(xmler.exec(s.restoreFile(o.colorMapFile(), &pal, XMLSciFigs::Data))) {
      _histogramLayer->setColorMap(pal);
    }
  }
  if(!o.appendLayerFiles().isEmpty()) {
    for(QStringList::const_iterator it=o.appendLayerFiles().begin(); it!=o.appendLayerFiles().end(); it++) {
      w->graphContents()->appendLayers(*it);
    }
  }
  if(!o.prependLayerFiles().isEmpty()) {
    for(QStringList::const_iterator it=o.prependLayerFiles().begin(); it!=o.prependLayerFiles().end(); it++) {
      w->graphContents()->prependLayers(*it);
    }
  }
  foreach(QString m, o.makeupObjectFiles()) {
    w->restoreMakeUp(m);
  }
  if(!o.preserveScales()) {
    setLimits();
  }
  if(!o.exportFile().isEmpty()) {
    w->exportFile(o.exportFile(), o.exportFormat(), o.dpi());
  }
}

void HistogramWidget::exportHistogramValues(const ExportOptions& o)
{
  TRACE;
  o.selectObjects(sheet->sheet());
  o.exportGridValues(sheet->sheet());
}

/*!

*/
void HistogramWidget::on_gridFilter_clicked()
{
  TRACE;
  Legend l;
  if(SampleClassification::run(curves, &l, _hist)) {
    setHistogram(_hist);
  }
}
