/***************************************************************************
**
**  This file is part of geopsy.
**
**  geopsy is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  geopsy is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-11-10
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <GeopsyCore.h>
#include <QGpGuiTools.h>

#include "Engine.h"
#include "MainWindow.h"
#include "TableWindow.h"
#include "SignalTableView.h"
#include "SetTableFields.h"
#include "ViewParamEditor.h"
#include "SigSelectionDnD.h"

TableWindow::TableWindow(QWidget* parent)
  : SubPoolWindow(parent)
{
  TRACE;
  setObjectName("table");
  setWindowIcon(QIcon(":table-22x22.png"));
  setMinimumSize(QSize(400,70));

  _headers=new SignalTableView(&subPool(), this);
  MainWindow * mw=WindowEnvironment::instance()->window(this);
  setEditionLocked(mw->isTableEditionLocked());

  // Drag and drop to transfert signal selection
  SigSelectionDnD * sigDnD=new SigSelectionDnD(_headers->viewport());
  connect(sigDnD, SIGNAL(selectionDropped(QWidget *, const SubSignalPool&)), this, SLOT(addSignals(QWidget *, const SubSignalPool&)));
  connect(sigDnD, SIGNAL(getSelection(SubSignalPool&)), this, SLOT(getSelection(SubSignalPool&)));
  _headers->installEventFilter(sigDnD);
  _headers->setDropIndicatorShown(true);

  _resizeCells.setInterval(1000);
  connect(&_resizeCells, SIGNAL(run()), this, SLOT(resizeCells()));
}

TableWindow::~TableWindow()
{
  TRACE;
}

void TableWindow::setEditionLocked(bool l)
{
  TRACE;
  if(l) {
    _headers->setEditTriggers(QAbstractItemView::NoEditTriggers);
    _headers->setSelectionBehavior(QAbstractItemView::SelectRows);
  } else {
    _headers->setEditTriggers(QAbstractItemView::AllEditTriggers);
    _headers->setSelectionBehavior(QAbstractItemView::SelectItems);
  }
}

QSize TableWindow::sizeHint() const
{
  TRACE;
  return QSize(700,400);
}

void TableWindow::setWindowTitle(QString title)
{
  TRACE;
  SubPoolWindow::setWindowTitle("Table - "+title);
}

void TableWindow::subPoolUpdate()
{
  TRACE;
  if(!subPool().isEmpty()) {
    setWindowTitle(subPool().name());
    endSignalChange();
  } else setWindowTitle(QString());
}

void TableWindow::getSelection(SubSignalPool& sel)
{
  TRACE;
  _headers->getSelection(sel);
  if(!subPool().name().contains(tr("Selected from"))) {
    sel.setName(tr("Selected from %1").arg(subPool().name()) );
  } else {
    sel.setName(subPool().name());
  }
}

void TableWindow::resizeEvent (QResizeEvent *)
{
  TRACE;
  _headers->setGeometry(0, 0, width(), height());
}

void TableWindow::exportTable ()
{
  TRACE;
  QString fileName=Message::getSaveFileName(tr("Export Table"), tr("Table (*.txt)"));
  if(fileName.length() > 0) {
    subPool().exportTable(fileName, _headers->dataFields());
  }
}

void TableWindow::importTable ()
{
  TRACE;
  QString fileName=Message::getOpenFileName(tr("Import Table"), tr("Table (*.txt)"));
  if(fileName.length()>0) {
    beginSignalChange();
    subPool().importTable(fileName);
    endSignalChange();
  }
}

void TableWindow::fastFourierTransform(DoubleSignal::SignalType st)
{
  TRACE;
  beginSignalChange();
  subPool().fastFourierTransform(st);
  endSignalChange();
}

void TableWindow::beginSignalChange(Signal * sig)
{
  TRACE;
  _headers->beginSignalChange(sig);
}

void TableWindow::endSignalChange(Signal * sig)
{
  TRACE;
  _headers->endSignalChange(sig);
  _resizeCells.newEvent();
}

void TableWindow::resizeCells()
{
  TRACE;
  _headers->resizeColumnsToContents();
  _headers->resizeRowsToContents();
}

void TableWindow::setDataFields()
{
  TRACE;
  SetTableFields * d=new SetTableFields(this);
  d->paramEditor->initTableFields(_headers->dataFields());
  Settings::getWidget(d);
  if(d->exec()==QDialog::Accepted) {
    switch(Message::question(MSG_ID, tr("Set table fields"), tr("Do you want to change also all new tables or just the current one?"),
                             tr("All new tables"), tr("Current table"), Message::cancel())) {
    case Message::Answer0: {
        GeopsyPreferences * prefs=GeopsyCoreEngine::instance()->preferences();
        d->paramEditor->getTableFields(prefs ->tableFields());
        prefs->saveNoAuto();
      }
      break;
    case Message::Answer1:
      break;
    case Message::Answer2:
      delete d;
      return;
    }
    Settings::setWidget(d);
    beginSignalChange();
    d->paramEditor->getTableFields(_headers->dataFields());
    endSignalChange();
  }
  delete d;
}
