/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-12-01
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef PEN_H
#define PEN_H

#include <QtGui>
#include <QGpCoreTools.h>
#include "SciFigsDLLExport.h"

namespace SciFigs {

class SCIFIGS_EXPORT Pen : public XMLClass
{
public:
  enum LineStyle {NoLine=0, SolidLine,
                  DashLine, Dash6Line, Dash8Line, DashSpace4Line, Dash6Space4Line, Dash8Space4Line,
                  DotLine, DotSpace4Line, DotSpace8Line,
                  DashDotLine, Dash6DotLine, Dash8DotLine,
                  DashDotDotLine, Dash6DotDotLine, Dash8DotDotLine,
                  LineStyleCount};
  enum CapStyle {SquareCap, FlatCap, RoundCap};
  enum JoinStyle {BevelJoin, MiterJoin, RoundJoin};

  ENUM_AS_STRING_DECL(LineStyle)
  ENUM_AS_STRING_DECL(CapStyle)
  ENUM_AS_STRING_DECL(JoinStyle)

  inline Pen();
  inline Pen(LineStyle style);
  inline Pen(const QColor& col, double width=0.3, LineStyle style=SolidLine);
  inline Pen(const QColor& col, double width, LineStyle s, CapStyle c, JoinStyle j);

  const QString& xml_tagName() const {return xmlPenTag;}
  static const QString xmlPenTag;

  QPen qpen(double dotpercm) const;
  inline bool operator==(const Pen& o) const;

  LineStyle lineStyle() const {return _lineStyle;}
  void setLineStyle(LineStyle s) {_lineStyle=s;}
  double width() const {return _width;}
  inline int screenWidth(double dotpercm) const;

  void setWidth (double w) {_width=w;}
  const QColor& color() const {return _color;}
  void setColor(const QColor & c) {_color=c;}
  CapStyle capStyle() const {return _capStyle;}
  void setCapStyle(CapStyle c) {_capStyle=c;}
  JoinStyle joinStyle() const {return _joinStyle;}
  void setJoinStyle(JoinStyle j) {_joinStyle=j;}

  static const Pen null;
protected:
  void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  bool xml_setProperty(XML_SETPROPERTY_ARGS);
  XMLMember xml_member(XML_MEMBER_ARGS);
private:
  QColor _color;
  LineStyle _lineStyle;
  CapStyle _capStyle;
  JoinStyle _joinStyle;
  // in mm
  double _width;
};

inline Pen::Pen()
{
  _color=Qt::black;
  _lineStyle=SolidLine;
  _capStyle=FlatCap;
  _joinStyle=BevelJoin;
  _width=0.3;
}

inline Pen::Pen(LineStyle style)
{
  _color=Qt::black;
  _lineStyle=style;
  _capStyle=FlatCap;
  _joinStyle=BevelJoin;
  _width=0.3;
}

inline Pen::Pen(const QColor& col, double width, LineStyle style)
{
  _color=col;
  _lineStyle=style;
  _capStyle=FlatCap;
  _joinStyle=BevelJoin;
  _width=width;
}

inline Pen::Pen(const QColor& col, double width, LineStyle s, CapStyle c, JoinStyle j)
{
  _color=col;
  _lineStyle=s;
  _capStyle=c;
  _joinStyle=j;
  _width=width;
}

inline bool Pen::operator==(const Pen& o) const
{
  TRACE;
  return _color==o._color
      && _lineStyle==o._lineStyle
      && _capStyle==o._capStyle
      && _joinStyle==o._joinStyle
      && _width==o._width;
}

inline int Pen::screenWidth(double dotpercm) const
{
  return qRound(0.1*_width*dotpercm);
}

} // namespace SciFigs

#endif // PEN_H
