/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-04-17
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef AXISWINDOW_H
#define AXISWINDOW_H

#include "SciFigsDLLExport.h"
#include "GraphicObject.h"
#include "GraphContents.h"

namespace SciFigs {

class GraphContentsLayer;

class SCIFIGS_EXPORT AxisWindow : public GraphicObject
{
  Q_OBJECT
  // Kept only to avoid warning on loading old files
  Q_PROPERTY(QString gridLines READ dummyPropertyString WRITE setDummyProperty STORED false)
  Q_PROPERTY(QString printLineWeight READ dummyPropertyString WRITE setDummyProperty STORED false)
  Q_PROPERTY(QString printBitmap READ dummyPropertyString WRITE setDummyProperty STORED false)
public:
  AxisWindow(QWidget * parent=0, Qt::WindowFlags f=Qt::Widget);
  ~AxisWindow();

  const QString& xml_tagName() const {return xmlAxisWindowTag;}
  static const QString xmlAxisWindowTag;

  void polish();
  Q_INVOKABLE GraphContents * graphContents() const {return _content;}
  Q_INVOKABLE Axis * xAxis() const {return _xAxis;}
  Q_INVOKABLE Axis * yAxis() const {return _yAxis;}

  // Updates all sub childs
  void updateScrollBars();

  void setHorizontalLine(int l);
  void setVerticalLine(int l);

  void swapAxes();
  void alignHScales(AxisWindow * ref, double atReal);
  void alignVScales(AxisWindow * ref, double atReal);

  virtual void addProperties(PropertyProxy * pp);
  virtual void removeProperties(PropertyProxy * pp);

  bool isZoomed() const {return !_zoomRects.isEmpty();}

  bool xml_polish(XML_POLISH_ARGS);

  static uint categoryXAxis, categoryYAxis;

  void scaleFonts(FontScales& orignal, double scale);
  void restoreScaleFonts(const FontScales& orignal);
  void print(QPainter& p, double dotpercm, int x0Sheet, int y0Sheet, bool mask);

  void setGeometry(double x, double y, double dx, double dy,
                   Axis::SizeType type=Axis::TotalSize);
  void addFreePicks();
public slots:
  void update();
  void deepUpdate();
  void zoomIn(double rx1, double ry1, double rx2, double ry2);
  void zoomOut();
  void browse(QRect& r);
  void setMapScale(double rx1, double ry1, double rx2, double ry2);
  void updateExternalGeometry();
  void updateInternalGeometry();
  virtual void updateGeometry();
  void checkZoomRects();
private slots:
  void xScrollChanged(int val);
  void yScrollChanged(int val);
protected:
  void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
  XMLMember xml_member(XML_MEMBER_ARGS);
private:
  void calcFromTotalSize(double dotpercm, double totalWidth,
                          double totalHeight, double& xWidth, double& yHeight,
                          QPainter * p=0) const;
  void calcSize(double dotpercm, double& xWidth, double& xHeight,
                 double& yWidth, double& yHeight,
                 double xOffset, double yOffset, QPainter * p=0) const;
  void calcScreenSize(double dotpercm, int& xWidth, int& yHeight,
                       int xOffset, int yOffset);
  void setAxisGeometry(double dotpercm);
  void setAxisGeometry(int xWidth, int yWidth, int xHeight, int yHeight);
  // Events Handlers
  void resizeEvent (QResizeEvent * e);
  void updateMask();
  // Calculates a common rounded scale from precise scale given by startAt
  double commonScale(double startAt);

  // Stack for last zoom rectangles, used by 'Unzoom'
  VectorList<Rect> _zoomRects;
  // Child widgets:
  Axis * _xAxis;
  Axis * _yAxis;
  QScrollBar * _xSBar;
  QScrollBar * _ySBar;
  GraphContents * _content;
};

} // namespace SciFigs

#endif // AXISWINDOW_H
